<?php
/**
 * Admin Functions
 *
 * @since       2.13.0
 *
 * @package     AffiliateWP
 * @subpackage  AffiliateWP\Admin
 * @copyright   Copyright (c) 2014, Sandhills Development, LLC
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! is_admin() ) {
	return; // Don't load any of these for e.g. the frontend.
}

/**
 * Icon Tooltip
 *
 * @since 2.13.0
 *
 * @param string $tooltip The tooltip to show on icon hover.
 * @param string $type    The type of info icon: normal, critical, warning.
 * @param bool   $echo    Whether to output (true) or return (false).
 * @param string $class   Custom color for the icon.
 *
 * @return string
 *
 * @throws \InvalidArgumentException If you do not use one of the qualified types.
 */
function affwp_icon_tooltip(
	string $tooltip = '',
	string $type = 'normal',
	bool $echo = true,
	string $class = ''
) : string {

	// See https://developer.wordpress.org/resource/dashicons/ for more icons.
	$dashicons_map = array(
		'critical' => 'dismiss',
		'normal'   => 'editor-help',
		'warning'  => 'warning',
		'setting'  => 'admin-settings',
		'global'   => 'admin-site',
		'unused'   => 'remove',
		'disabled' => 'remove',
		'hidden'   => 'hidden',
		'mdash'    => 'minus',
	);

	if ( ! in_array( $type, array_keys( $dashicons_map ), true ) ) {
		throw new \InvalidArgumentException( '$type can only be one of: normal, warning, critical.' );
	}

	$id = wp_unique_id( wp_rand() );

	ob_start();

	?>
			<span
				class="affwp-tooltip icon alpine <?php echo esc_attr( $type ); ?> tooltip-<?php echo esc_attr( $id ); ?> <?php echo esc_attr( $class ); ?>"
				aria-describedby="tooltip-<?php echo esc_attr( $id ); ?>"
				x-data='{ "tooltip": [] }'><!--

					--><span
							id="tooltip-<?php echo esc_attr( $id ); ?>"
							class="dashicon dashicons dashicons-<?php echo esc_attr( $dashicons_map[ $type ] ); ?>"
							x-on:mouseover="tooltip.showTooltip<?php echo esc_attr( $id ); ?> = true;"
							x-on:mouseover.away="tooltip.showTooltip<?php echo esc_attr( $id ); ?> = false;"><!--

								--><template x-if="tooltip.showTooltip<?php echo esc_attr( $id ); ?>"><span
										class="affwp-tooltip-content active"
										x-show="tooltip.showTooltip<?php echo esc_attr( $id ); ?>"><?php echo esc_html( $tooltip ); ?></span></template><!--
						--></span><!--
			--></span>

	<?php

	if ( $echo ) {

		// Echo instead.
		echo wp_kses(
			trim( ob_get_clean() ),
			affwp_get_tooltip_allowed_html()
		);

		return '';
	}

	return trim( ob_get_clean() );
}

/**
 * Text Tooltip
 *
 * @since 2.13.0
 *
 * @param string $text    The text to initiate the tooltip.
 * @param string $tooltip The tooltip contents.
 * @param bool   $echo    Set to true to echo.
 *
 * @return [type] [description]
 */
function affwp_text_tooltip( string $text, string $tooltip, bool $echo = true ) {

	$id = wp_unique_id( wp_rand() );

	ob_start();

	?>

	<span
		x-data='{ "tooltip": [] }'
		class="trigger affwp-tooltip text alpine"
		x-on:mouseover="tooltip.showTooltip<?php echo esc_attr( $id ); ?> = true;"
		x-on:mouseover.away="tooltip.showTooltip<?php echo esc_attr( $id ); ?> = false;"
		aria-describedby="tooltip-<?php echo esc_attr( $id ); ?>"><!--

			--><?php echo wp_kses_post( $text ); ?><!--

		--><template x-if="tooltip.showTooltip<?php echo esc_attr( $id ); ?>"><span
				id="tooltip-<?php echo esc_attr( $id ); ?>"
				class="affwp-tooltip-content active text"
				x-show="tooltip.showTooltip<?php echo esc_attr( $id ); ?>"><!--

					--><?php echo wp_kses_post( $tooltip ); ?><!--

			--></span></template><!--

	--></span>

	<?php

	if ( $echo ) {

		// Echo instead.
		echo wp_kses_post(
			trim( ob_get_clean() ),
		);

		return '';
	}

	return trim( ob_get_clean() );
}

/**
 * Tooltip Allowed HTML.
 *
 * @since 2.13.0
 *
 * @return array
 */
function affwp_get_tooltip_allowed_html() : array {

	return array_merge(
		array(
			'span' => array(
				'x-data'              => true,
				'id'                  => true,
				'class'               => true,
				'x-transition'        => true,
				'x-show'              => true,
				'aria-describedby'    => true,
				'x-on:mouseover'      => true,
				'x-on:mouseover.away' => true,
				'style'               => true,
			),
		)
	);
}

/**
 * Load tooltip CSS/JS on our admin pages.
 *
 * @since 2.13.0
 *
 * @return void
 */
function affwp_alpine_all_admin_pages() : void {

	if ( ! affwp_is_admin_page() ) {
		return;
	}

	wp_enqueue_script( 'alpinejs' );
}
add_action( 'admin_enqueue_scripts', 'affwp_alpine_all_admin_pages' );
