<?php
ob_start();
class Subscribe_Form_With_Integration_Forms_List_Table extends WP_List_Table {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The table name in database of the forms.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $table_name    The table name in database of the forms.
     */
    private $table_name;

    /**
     * The settings object of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      object    $settings_obj    The settings object of this plugin.
     */
    private $settings_obj;


    private $title_length;


    /** Class constructor */
    public function __construct( $plugin_name ) {
        global $wpdb;

        $this->plugin_name = $plugin_name;

        $this->table_name = $wpdb->prefix . "sf_forms";

        parent::__construct( array(
            'singular' => __( 'Form', $this->plugin_name ), //singular name of the listed records
            'plural'   => __( 'Forms', $this->plugin_name ), //plural name of the listed records
            'ajax'     => false //does this table support ajax?
        ) );

        add_action( 'admin_notices', array( $this, 'sf_forms_notices' ) );
        add_filter( 'default_hidden_columns', array( $this, 'get_hidden_columns'), 10, 2 );

    }

    /**
     * Override of table nav to avoid breaking with bulk actions & according nonce field
     */
    public function display_tablenav( $which ) {
        ?>
        <div class="tablenav <?php echo esc_attr( $which ); ?>">
            
            <div class="alignleft actions">
                <?php  $this->bulk_actions( $which ); ?>
            </div>
            <?php
            $this->extra_tablenav( $which );
            $this->pagination( $which );
            ?>
            <br class="clear" />
        </div>
        <?php
    }

    /**
     * Disables the views for 'side' context as there's not enough free space in the UI
     * Only displays them on screen/browser refresh. Else we'd have to do this via an AJAX DB update.
     *
     * @see WP_List_Table::extra_tablenav()
     */
    public function extra_tablenavasd( $which ) {
        global $wpdb;
        $titles_sql = "SELECT s.title, s.id
                       FROM " .$wpdb->prefix . "form_categories AS s
                       WHERE s.status = 'published'";
        $cat_titles = $wpdb->get_results($titles_sql);

        $cat_id = null;
        if( isset( $_GET['filterby'] )){
            $cat_id = absint( sanitize_text_field( $_GET['filterby'] ) );
        }
        $categories_select = array();
        foreach($cat_titles as $key => $cat_title){
            $selected = "";
            if($cat_id === intval($cat_title->id)){
                $selected = "selected";
            }
            $categories_select[$cat_title->id]['title'] = $cat_title->title;
            $categories_select[$cat_title->id]['selected'] = $selected;
            $categories_select[$cat_title->id]['id'] = $cat_title->id;
        }
        sort($categories_select);
        ?>
        <div id="category-filter-div-formlist" class="alignleft actions bulkactions">
            <select name="filterby-<?php echo esc_attr( $which ); ?>" id="form-category-filter-<?php echo esc_attr( $which ); ?>">
                <option value=""><?php echo __('Select Category',$this->plugin_name)?></option>
                <?php
                    foreach($categories_select as $key => $cat_title){
                        echo "<option ".$cat_title['selected']." value='".$cat_title['id']."'>".$cat_title['title']."</option>";
                    }
                ?>
            </select>
            <input type="button" id="doaction-<?php echo esc_attr( $which ); ?>" class="cat-filter-apply-<?php echo esc_attr( $which ); ?> button" value="Filter">
        </div>
        
        <a style="margin: 0px 8px 0 0;" href="?page=<?php echo esc_attr( sanitize_text_field( $_REQUEST['page'] ) ); ?>" class="button"><?php echo __( "Clear filters", $this->plugin_name ); ?></a>
        <?php
    }
    
    protected function get_views() {
        $published_count = $this->get_statused_record_count( 'published' );
        $draft_count = $this->get_statused_record_count( 'draft' );
        $trashed_count = $this->get_statused_record_count( 'trashed' );
        $all_count = $this->all_record_count();
        $selected_all = "";
        $selected_published = "";
        $selected_draft = "";
        $selected_trashed = "";
        if( isset( $_GET['fstatus'] ) ){
            switch( sanitize_text_field( $_GET['fstatus'] ) ){
                case "published":
                    $selected_published = " style='font-weight:bold;' ";
                    break;
                case "draft":
                    $selected_draft = " style='font-weight:bold;' ";
                    break;
                case "trashed":
                    $selected_trashed = " style='font-weight:bold;' ";
                    break;
                default:
                    $selected_all = " style='font-weight:bold;' ";
                    break;
            }
        }else{
            $selected_all = " style='font-weight:bold;' ";
        }
        $status_links = array(
            "all" => "<a ".$selected_all." href='?page=".esc_attr( $_REQUEST['page'] )."'>" . __( "All", $this->plugin_name ) . " (".$all_count.")</a>",
        );
        if( intval( $published_count ) > 0 ){
            $status_links["published"] = "<a ".$selected_published." href='?page=".esc_attr( $_REQUEST['page'] )."&fstatus=published'>" . __( "Published", $this->plugin_name ) . " (".$published_count.")</a>";
        }
        if( intval( $draft_count ) > 0 ){
            $status_links["draft"] = "<a ".$selected_draft." href='?page=".esc_attr( $_REQUEST['page'] )."&fstatus=draft'>" . __( "Draft", $this->plugin_name ) . " (".$draft_count.")</a>";
        }
        if( intval( $trashed_count ) > 0 ){
            $status_links["trashed"] = "<a ".$selected_trashed." href='?page=".esc_attr( $_REQUEST['page'] )."&fstatus=trashed'>" . __( "Trash", $this->plugin_name ) . " (".$trashed_count.")</a>";
        }
        return $status_links;
    }

    /**
     * Retrieve customers data from the database
     *
     * @param int $per_page
     * @param int $page_number
     *
     * @return mixed
     */
    public static function get_items( $per_page = 20, $page_number = 1, $search = '' ) {

        global $wpdb;
        
        $sql = "SELECT * FROM ". $wpdb->prefix . "sf_forms";
        
        $where = array();

        if ( isset( $_GET['fstatus'] ) && $_GET['fstatus'] != ''){
            $where[] = ' status = "' . esc_sql( sanitize_text_field( $_GET['fstatus'] ) ) . '" ';
        }else{
            $where[] = ' status != "trashed" ';
        }

        if( $search != '' ){
            $where[] = $search;
        }

        if ( ! empty( $where ) ){
            $sql .= ' WHERE ' . implode( ' AND ', $where );
        }

        if ( ! empty( $_REQUEST['orderby'] ) ) {
            $order_by  = ( isset( $_REQUEST['orderby'] ) && sanitize_text_field( $_REQUEST['orderby'] ) != '' ) ? sanitize_text_field( $_REQUEST['orderby'] ) : 'id';
            $order_by .= ( ! empty( $_REQUEST['order'] ) && strtolower( $_REQUEST['order'] ) == 'asc' ) ? ' ASC' : ' DESC';

            $sql_orderby = sanitize_sql_orderby( $order_by );

            if ( $sql_orderby ) {
                $sql .= ' ORDER BY ' . $sql_orderby;
            } else {
                $sql .= ' ORDER BY id DESC';
            }
        }else{
            $sql .= ' ORDER BY id DESC';
        }

        $sql .= " LIMIT $per_page";
        $sql .= ' OFFSET ' . ( $page_number - 1 ) * $per_page;


        $result = $wpdb->get_results( $sql, 'ARRAY_A' );

        return $result;
    }

    public static function get_item_by_id( $id ){
        global $wpdb;

        $sql = "SELECT * FROM " . $wpdb->prefix . "sf_forms WHERE id=" . absint( $id );

        $result = $wpdb->get_row($sql, 'ARRAY_A');

        return $result;
    }

    public function add_or_edit_item(){
        global $wpdb;

        if( isset( $_POST["sf_form_action"] ) && wp_verify_nonce( $_POST["sf_form_action"], 'sf_form_action' ) ){

            $name_prefix = 'sf_form_';
            
            // Save type
            $save_type = (isset($_POST['save_type'])) ? $_POST['save_type'] : '';

            // Id of item
            $id = isset( $_POST['id'] ) ? absint( $_POST['id'] ) : 0;
            
            $form = null;
            if( $id != 0 ){
                $form = $this->get_item_by_id( $id );
            }
            
            // Title
            $title = isset( $_POST[ $name_prefix . 'title' ] ) && $_POST[ $name_prefix . 'title' ] != '' ? stripslashes( sanitize_text_field( $_POST[ $name_prefix . 'title' ] ) ) : '';

            if($title == ''){
                $url = esc_url_raw( remove_query_arg( false ) );
                wp_redirect( $url );
            }

            // Description
            $description = isset( $_POST[ $name_prefix . 'description' ] ) && $_POST[ $name_prefix . 'description' ] != '' ? stripslashes( $_POST[ $name_prefix . 'description' ] ) : '';

            // Status
            $status = isset( $_POST[ $name_prefix . 'status' ] ) && $_POST[ $name_prefix . 'status' ] != '' ? sanitize_text_field( $_POST[ $name_prefix . 'status' ] ) : 'published';

            // Form fields
            $form_fields = array(
                'labels' => isset( $_POST[ $name_prefix . 'fields_labels'] ) && !empty( $_POST[ $name_prefix . 'fields_labels' ] ) ? $_POST[ $name_prefix . 'fields_labels' ] : array(),
            );
            
            // Form fields active
            $form_fields_active = array();
            $form_fields_active_post = isset( $_POST[ $name_prefix . 'fields_active'] ) && !empty( $_POST[ $name_prefix . 'fields_active' ] ) ? $_POST[ $name_prefix . 'fields_active' ] : array();
            $form_fields_active['fullname'] = isset( $form_fields_active_post['fullname'] ) && $form_fields_active_post['fullname'] == 'on' ? 'on' : 'off';
            $form_fields_active['phone'] = isset( $form_fields_active_post['phone'] ) && $form_fields_active_post['phone'] == 'on' ? 'on' : 'off';
            $form_fields_active['email'] = isset( $form_fields_active_post['email'] ) && $form_fields_active_post['email'] == 'on' ? 'on' : 'off';
            
            
            // Submit button
            $submit_button = array(
                'text' => isset( $_POST[ $name_prefix . 'submit_button_text'] ) && $_POST[ $name_prefix . 'submit_button_text' ] != '' ? sanitize_text_field( $_POST[ $name_prefix . 'submit_button_text' ] ) : '',
            );
            
            $message_after_subscribtion = isset( $_POST[ $name_prefix . 'message_after_subscribtion'] ) && $_POST[ $name_prefix . 'message_after_subscribtion' ] != '' ? sanitize_text_field( $_POST[ $name_prefix . 'message_after_subscribtion' ] ) : '';
            
            $form_content = isset( $_POST[ $name_prefix . 'form_content' ] ) && $_POST[ $name_prefix . 'form_content' ] != '' ? stripslashes( $_POST[ $name_prefix . 'form_content'] ) : '';
            $content_bg_image = isset( $_POST[ $name_prefix . 'content_bg_image' ] ) && $_POST[ $name_prefix . 'content_bg_image' ] != '' ? sanitize_text_field( $_POST[ $name_prefix . 'content_bg_image'] ) : '';

            
            // Options
            $options = array(
                'message_after_subscribtion' => $message_after_subscribtion,
                'form_content' => $form_content,
                'content_bg_image' => $content_bg_image,
            );
            
            $enable_google_sheets = "off";
            if( isset( $_POST[ $name_prefix . 'enable_google_sheet'] ) && $_POST[ $name_prefix . 'enable_google_sheet'] == "on" ){
                $enable_google_sheets = "on";
                $sheet_id = SF_Google_Integration_Methods::get_sheet_id($this_id);
                $old_sheet_id         = $sheet_id;
                $check_sheet_id       = $sheet_id !== null ? true : false;
                
                $google = get_option('sf_google_integration', array());
                $google_client        = isset($google['client']) ? $google['client'] : '';
                $google_secret        = isset($google['secret']) ? $google['secret'] : '';
                $google_token         = isset($google['token']) ? $google['token'] : '';
                $google_refresh_token = isset($google['refresh_token']) ? $google['refresh_token'] : '';
                
                $this_form_title = $title;

                $spreadsheet_id = '';

                $google_sheet_data = array(
                    "refresh_token" => $google_refresh_token,
                    "client_id"     => $google_client,
                    "client_secret" => $google_secret,
                    "form_title"    => $this_form_title,
                    "sheet_id"      => $old_sheet_id,
                    'id'            => $this_id
                );
                
                if(!$check_sheet_id){
                    $spreadsheet_id = SF_Google_Integration_Methods::get_google_sheet_id($google_sheet_data);                    
                }
                else{
                    if($old_sheet_id != ''){
                        $spreadsheet_id = $old_sheet_id;
                        SF_Google_Integration_Methods::update_google_spreadsheet( $google_sheet_data );
                    }
                }
                $options['spreadsheet_id'] = $spreadsheet_id;
                
            }
            $options['enable_google_sheet'] = $enable_google_sheets;
            
            // Klaviyo integration
            $options['enable_klaviyo'] = isset( $_POST[ $name_prefix . 'enable_klaviyo'] ) && $_POST[ $name_prefix . 'enable_klaviyo' ] == 'on' ? 'on' : 'off';
            $options['klaviyo_list'] = isset( $_POST[ $name_prefix . 'klaviyo_list'] ) && $_POST[ $name_prefix . 'klaviyo_list' ] != '' ? sanitize_text_field( $_POST[ $name_prefix . 'klaviyo_list' ] ) : '';

            $message = '';
            if( $id == 0 ){
                $result = $wpdb->insert(
                    $this->table_name,
                    array(
                        'form_title'        => $title,
                        'form_description'  => $description,
                        'status'            => $status,
                        'date_created'      => current_time( 'mysql' ),
                        'form_fields'       => json_encode( $form_fields ),
                        'form_fields_active'=> json_encode( $form_fields_active ),
                        'submit_button'     => json_encode( $submit_button ),
                        'options'           => json_encode( $options ),
                    ),
                    array(
                        '%s', // form_title
                        '%s', // form_description
                        '%s', // status
                        '%s', // date_created
                        '%s', // date_modified
                        '%s', // form_fields
                        '%s', // form_fields_active
                        '%s', // submit_button
                        '%s', // options
                    )
                );

                $inserted_id = $wpdb->insert_id;
                $message = 'created';
            }else{
                $result = $wpdb->update(
                    $this->table_name,
                    array(
                        'form_title'        => $title,
                        'form_description'  => $description,
                        'status'            => $status,
                        'form_fields'       => json_encode( $form_fields ),
                        'form_fields_active'=> json_encode( $form_fields_active ),
                        'submit_button'     => json_encode( $submit_button ),
                        'options'           => json_encode( $options ),
                    ),
                    array( 'id' => $id ),
                    array(
                        '%s', // form_title
                        '%s', // form_description
                        '%s', // status
                        '%s', // form_fields
                        '%s', // form_fields_active
                        '%s', // submit_button
                        '%s', // options
                    ),
                    array( '%d' )
                );

                $inserted_id = $id;
                $message = 'updated';
            }

            if( $result >= 0  ) {
                if($save_type == 'apply'){
                    if($id == 0){
                        $url = esc_url_raw( add_query_arg( array(
                            "action"    => "edit",
                            "id"        => $inserted_id,
                            "status"    => $message
                        ) ) );
                    }else{
                        $url = esc_url_raw( add_query_arg( array(
                            "status" => $message
                        ) ) );
                    }
                    wp_redirect( $url );
                }elseif($save_type == 'save_new'){
                    $url = remove_query_arg( array('id') );
                    $url = esc_url_raw( add_query_arg( array(
                        "action" => "add",
                        "status" => $message
                    ), $url ) );
                    wp_redirect( $url );
                }else{
                    $url = remove_query_arg( array('action', 'id') );
                    $url = esc_url_raw( add_query_arg( array(
                        "status" => $message
                    ), $url ) );
                    wp_redirect( $url );
                }
            }
        }
    }
    
    private function get_max_id() {
        global $wpdb;
        $table = $wpdb->prefix . "sf_forms";

        $sql = "SELECT MAX(id) FROM {$table}";

        $result = $wpdb->get_var($sql);

        return $result;
    }

    /**
     * Delete a customer record.
     *
     * @param int $id customer ID
     */
    public static function delete_items( $id ) {
        global $wpdb;

        $wpdb->delete(
            $wpdb->prefix . "sf_forms",
            array( 'id' => absint( $id ) ),
            array( '%d' )
        );

    }

    /**
     * Move to trash a customer record.
     *
     * @param int $id customer ID
     */
    public static function trash_items( $id ) {
        global $wpdb;
        $db_item = self::get_item_by_id( $id );

        $wpdb->update(
            $wpdb->prefix . "sf_forms",
            array( 
                'status' => 'trashed',
                'trash_status' => $db_item['status'],
            ),
            array( 'id' => absint( $id ) ),
            array( '%s', '%s' ),
            array( '%d' )
        );

    }

    /**
     * Restore a customer record.
     *
     * @param int $id customer ID
     */
    public static function restore_items( $id ) {
        global $wpdb;
        $db_item = self::get_item_by_id( $id );

        $wpdb->update(
            $wpdb->prefix . "sf_forms",
            array( 
                'status' => $db_item['trash_status'],
                'trash_status' => '',
            ),
            array( 'id' => absint( $id ) ),
            array( '%s', '%s' ),
            array( '%d' )
        );
    }

    /**
     * Duplicate a customer record.
     *
     * @param int $id customer ID
     */
    public function duplicate_items( $id ){
        global $wpdb;
        
        $object = $this->get_item_by_id( $id );
        
        $author_id = get_current_user_id();
        
        $options = json_decode($object['options'], true);
        
        $result = $wpdb->insert(
            $this->table_name,
            array(
                'form_title'            => $object['form_title'] . " (copy)",
                'form_description'      => $object['form_description'],
                'form_fields'           => $object['form_fields'],
                'form_fields_active'    => $object['form_fields_active'],
                'submit_button'         => $object['submit_button'],
                'date_created'          => current_time( 'mysql' ),
                'status'                => $object['status'],
                'options'               => json_encode( $options, JSON_UNESCAPED_SLASHES ),
            ),
            array(
                '%s', // form_title
                '%s', // form_description
                '%s', // form_fields
                '%s', // form_fields_active
                '%s', // submit_button
                '%s', // date_created
                '%s', // status
                '%s', // options
            )
        );
    }


    /**
     * Returns the count of records in the database.
     *
     * @return null|string
     */
    public static function record_count() {
        global $wpdb;
        $filter = array();
        $sql = "SELECT COUNT(*) FROM " . $wpdb->prefix . "sf_forms";
        
        if( isset( $_GET['filterby'] ) && intval($_GET['filterby']) > 0){
            $cat_id = intval( sanitize_text_field( $_GET['filterby'] ) );
            $filter[] = ' FIND_IN_SET('.$cat_id.',category_ids) ';
        }
        
        if( isset( $_REQUEST['fstatus'] ) ){
            $fstatus = sanitize_text_field( $_REQUEST['fstatus'] );
            if($fstatus !== null){
                $filter[] = " status = '". esc_sql( $fstatus ) ."' ";
            }
        }else{
            $filter[] = " status != 'trashed' ";
        }
        
        if(count($filter) !== 0){
            $sql .= " WHERE ".implode(" AND ", $filter);
        }

        return $wpdb->get_var( $sql );
    }
    
    public static function all_record_count() {
        global $wpdb;

        $sql = "SELECT COUNT(*) FROM " . $wpdb->prefix . "sf_forms WHERE status != 'trashed'";

        if( isset( $_GET['filterby'] ) && intval($_GET['filterby']) > 0){
            $cat_id = intval( sanitize_text_field( $_GET['filterby'] ) );
            $sql .= ' AND '.$cat_id.' IN (category_ids) ';
        }

        return $wpdb->get_var( $sql );
    }

    public static function published_questions_record_count() {
        global $wpdb;

        $sql = "SELECT COUNT(*) FROM " . $wpdb->prefix . "questions WHERE status = 'published'";

        return $wpdb->get_var( $sql );
    }

    public static function get_statused_record_count( $status ) {
        global $wpdb;

        $sql = "SELECT COUNT(*) FROM " . $wpdb->prefix . "sf_forms WHERE status='" . esc_sql( $status ) . "'";

        if( isset( $_GET['filterby'] ) && intval($_GET['filterby']) > 0){
            $cat_id = intval( sanitize_text_field( $_GET['filterby'] ) );
            $sql .= ' AND '.$cat_id.' IN (category_ids) ';
        }

        return $wpdb->get_var( $sql );
    }

    public static function get_passed_users_count( $id ) {
        global $wpdb;
        $id = absint($id);
        $sql = "SELECT COUNT(*) FROM " . $wpdb->prefix . "submissions WHERE form_id=".$id;

        return $wpdb->get_var( $sql );
    }


    /** Text displayed when no customer data is available */
    public function no_items() {
        echo __( 'There are no forms yet.', $this->plugin_name );
    }


    /**
     * Render a column when no column specific method exist.
     *
     * @param array $item
     * @param string $column_name
     *
     * @return mixed
     */
    public function column_default( $item, $column_name ) {
        switch ( $column_name ) {
            case 'title':
            case 'items_count':
            case 'status':
            case 'id':
                return $item[ $column_name ];
                break;
            default:
                return print_r( $item, true ); //Show the whole array for troubleshooting purposes
        }
    }

    /**
     * Render the bulk edit checkbox
     *
     * @param array $item
     *
     * @return string
     */
    function column_cb( $item ) {
        return sprintf(
            '<input type="checkbox" name="bulk-delete[]" value="%s" />', $item['id']
        );
    }


    /**
     * Method for name column
     *
     * @param array $item an array of DB data
     *
     * @return string
     */
    function column_title( $item ) {
        if($item['status'] == 'trashed'){
            $delete_nonce = wp_create_nonce( $this->plugin_name . '-delete-form' );
        }else{
            $delete_nonce = wp_create_nonce( $this->plugin_name . '-trash-form' );
        }

        $form_title = stripcslashes( $item['form_title'] );

        $q = esc_attr( $form_title );
        
        $fstatus = '';
        if( isset( $_GET['fstatus'] ) && $_GET['fstatus'] != '' ){
            $fstatus = '&fstatus=' . sanitize_text_field( $_GET['fstatus'] );
        }

        $title = sprintf( '<a href="?page=%s&action=%s&id=%d" title="%s">%s</a>', esc_attr( $_REQUEST['page'] ), 'edit', absint( $item['id'] ), $q, $form_title );

        $actions = array();
        if($item['status'] == 'trashed'){
            $title = sprintf( '<a><strong>%s</strong></a>', $form_title );
            $actions['restore'] = sprintf( '<a href="?page=%s&action=%s&id=%d&_wpnonce=%s'.$fstatus.'">'. __('Restore', $this->plugin_name) .'</a>', esc_attr( $_REQUEST['page'] ), 'restore', absint( $item['id'] ), $delete_nonce );
            $actions['delete'] = sprintf( '<a class="ays_confirm_del" data-message="%s" href="?page=%s&action=%s&id=%s&_wpnonce=%s'.$fstatus.'">'. __('Delete Permanently', $this->plugin_name) .'</a>', $form_title, esc_attr( $_REQUEST['page'] ), 'delete', absint( $item['id'] ), $delete_nonce );
        }else{
            $draft_text = '';
            if( $item['status'] == 'draft' && !( isset( $_GET['fstatus'] ) && $_GET['fstatus'] == 'draft' )){
                $draft_text = ' — ' . '<span class="post-state">' . __( "Draft", $this->plugin_name ) . '</span>';
            }
            $title = sprintf( '<strong><a href="?page=%s&action=%s&id=%d" title="%s">%s</a>%s</strong>', esc_attr( $_REQUEST['page'] ), 'edit', absint( $item['id'] ), $q, $form_title, $draft_text );

            $actions['edit'] = sprintf( '<a href="?page=%s&action=%s&id=%d">'. __('Edit', $this->plugin_name) .'</a>', esc_attr( $_REQUEST['page'] ), 'edit', absint( $item['id'] ) );

            $actions['duplicate'] = sprintf( '<a href="?page=%s&action=%s&id=%d&_wpnonce=%s'.$fstatus.'">'. __('Duplicate', $this->plugin_name) .'</a>', esc_attr( $_REQUEST['page'] ), 'duplicate', absint( $item['id'] ), $delete_nonce );
            $actions['trash'] = sprintf( '<a href="?page=%s&action=%s&id=%s&_wpnonce=%s'.$fstatus.'">'. __('Move to trash', $this->plugin_name) .'</a>', esc_attr( $_REQUEST['page'] ), 'trash', absint( $item['id'] ), $delete_nonce );
        }

        return $title . $this->row_actions( $actions );
    }

    function column_status( $item ) {
        global $wpdb;
        $status = ucfirst( $item['status'] );
        $date = date( 'Y/m/d', strtotime( $item['date_created'] ) );
        $title_date = date( 'l jS \of F Y h:i:s A', strtotime( $item['date_created'] ) );
        $html = "<p style='font-size:14px;margin:0;'>" . $status . "</p>";
        $html .= "<p style=';font-size:14px;margin:0;text-decoration: dotted underline;' title='" . $title_date . "'>" . $date . "</p>";
        return $html;
    }

    function column_items_count( $item ) {
        global $wpdb;
        $sql = "SELECT COUNT(*) FROM ". $wpdb->prefix ."sf_subscribers WHERE form_id = ". $item['id'];
        $result = $wpdb->get_var( $sql );
        if( !$result ){
            $count = 0;
        }else{
            $count = intval( $result );
        }
        return "<p style='font-size:14px;'>" . $count . "</p>";
    }
    
    function column_shortcode( $item ) {
        $shortcode = htmlentities('[sf_popup_form id="'.$item["id"].'"]');
        return '<input type="text" onClick="this.setSelectionRange(0, this.value.length)" readonly value="'.$shortcode.'" />';
    }

    /**
     *  Associative array of columns
     *
     * @return array
     */
    function get_columns() {
        $columns = array(
            'cb' => '<input type="checkbox" />',
            'title' => __( 'Title', $this->plugin_name ),
        );

        $columns['items_count'] = __( 'Subscribers count', $this->plugin_name );
        $columns['shortcode'] = __( 'Shortcode', $this->plugin_name );
        $columns['status'] = __( 'Status', $this->plugin_name );
        $columns['id'] = __( 'ID', $this->plugin_name );

        if( isset( $_GET['action'] ) && ( $_GET['action'] == 'add' || $_GET['action'] == 'edit' ) ){
            return array();
        }
        
        return $columns;
    }


    /**
     * Columns to make sortable.
     *
     * @return array
     */
    public function get_sortable_columns() {
        $sortable_columns = array(
            'title'         => array( 'title', true ),
            'id'            => array( 'id', true ),
        );

        return $sortable_columns;
    }

    /**
     * Columns to make hidden.
     *
     * @return array
     */
    public function get_hidden_columns() {
        $sortable_columns = array(
            'category_ids',
            'code_include',
        );

        return $sortable_columns;
    }

    /**
     * Returns an associative array containing the bulk action
     *
     * @return array
     */
    public function get_bulk_actions() {
        $actions = array(
            // 'bulk-duplicate' => __( 'Duplicate', $this->plugin_name ),
            'bulk-trash' => __( 'Move to trash', $this->plugin_name ),
        );

        if(isset($_GET['fstatus']) && sanitize_text_field( $_GET['fstatus'] ) == 'trashed'){
            $actions = array(
                'bulk-restore' => __( 'Restore', $this->plugin_name ),
                'bulk-delete' => __( 'Delete Permanently', $this->plugin_name ),
            );
        }

        return $actions;
    }



    /**
     * Handles data query and filter, sorting, and pagination.
     */
    public function prepare_items() {
        global $wpdb;
        $this->_column_headers = $this->get_column_info();

        /** Process bulk action */
        $this->process_bulk_action();

        $per_page     = $this->get_items_per_page( 'forms_per_page', 20 );
        $current_page = $this->get_pagenum();
        $total_items  = self::record_count();

        $this->set_pagination_args( array(
            'total_items' => $total_items, //WE have to calculate the total number of items
            'per_page'    => $per_page, //WE have to determine how many items to show on a page
            'total_pages' => ceil( $total_items / $per_page )
        ) );

        $search = ( isset( $_REQUEST['s'] ) ) ? sanitize_text_field( $_REQUEST['s'] ) : false;

        $do_search = ( $search ) ? sprintf( " title LIKE '%%%s%%' ", esc_sql( $wpdb->esc_like( $search ) ) ) : '';

        $this->items = self::get_items( $per_page, $current_page, $do_search );
    }

    public function process_bulk_action() {
       
        //Detect when a bulk action is being triggered...
        if ( 'delete' === $this->current_action() ) {

            // In our file that handles the request, verify the nonce.
            $nonce = esc_attr( $_REQUEST['_wpnonce'] );

            if ( ! wp_verify_nonce( $nonce, $this->plugin_name . '-delete-form' ) ) {
                die( 'Go get a life script kiddies' );
            }
            else {
                self::delete_items( absint( $_GET['id'] ) );

                // esc_url_raw() is used to prevent converting ampersand in url to "#038;"
                // add_query_arg() return the current url

                $add_query_args = array(
                    "status" => 'deleted'
                );
                if( isset( $_GET['fstatus'] ) && $_GET['fstatus'] != '' ){
                    $add_query_args['fstatus'] = sanitize_text_field( $_GET['fstatus'] );
                }
                $url = remove_query_arg( array('action', 'id', '_wpnonce') );
                $url = esc_url_raw( add_query_arg( $add_query_args, $url ) );
                wp_redirect( $url );
            }

        }

        //Detect when a bulk action is being triggered...
        if ( 'trash' === $this->current_action() ) {

            // In our file that handles the request, verify the nonce.
            $nonce = esc_attr( $_REQUEST['_wpnonce'] );

            if ( ! wp_verify_nonce( $nonce, $this->plugin_name . '-trash-form' ) ) {
                die( 'Go get a life script kiddies' );
            }
            else {
                self::trash_items( absint( $_GET['id'] ) );

                // esc_url_raw() is used to prevent converting ampersand in url to "#038;"
                // add_query_arg() return the current url

                $add_query_args = array(
                    "status" => 'trashed'
                );
                if( isset( $_GET['fstatus'] ) && $_GET['fstatus'] != '' ){
                    $add_query_args['fstatus'] = sanitize_text_field( $_GET['fstatus'] );
                }
                $url = remove_query_arg( array('action', 'id', '_wpnonce') );
                $url = esc_url_raw( add_query_arg( $add_query_args, $url ) );
                wp_redirect( $url );
            }

        }

        //Detect when a bulk action is being triggered...
        if ( 'restore' === $this->current_action() ) {

            // In our file that handles the request, verify the nonce.
            $nonce = esc_attr( $_REQUEST['_wpnonce'] );

            if ( ! wp_verify_nonce( $nonce, $this->plugin_name . '-delete-form' ) ) {
                die( 'Go get a life script kiddies' );
            }
            else {
                self::restore_items( absint( $_GET['id'] ) );

                // esc_url_raw() is used to prevent converting ampersand in url to "#038;"
                // add_query_arg() return the current url

                $add_query_args = array(
                    "status" => 'restored'
                );
                if( isset( $_GET['fstatus'] ) && $_GET['fstatus'] != '' ){
                    $add_query_args['fstatus'] = sanitize_text_field( $_GET['fstatus'] );
                }
                $url = remove_query_arg( array('action', 'id', '_wpnonce') );
                $url = esc_url_raw( add_query_arg( $add_query_args, $url ) );
                wp_redirect( $url );
            }

        }

        //Detect when a bulk action is being triggered...
        if ( 'duplicate' === $this->current_action() ) {

            // In our file that handles the request, verify the nonce.
            $nonce = esc_attr( $_REQUEST['_wpnonce'] );

            if ( ! wp_verify_nonce( $nonce, $this->plugin_name . '-trash-form' ) ) {
                die( 'Go get a life script kiddies' );
            }
            else {
                $this->duplicate_items( absint( $_GET['id'] ) );

                // esc_url_raw() is used to prevent converting ampersand in url to "#038;"
                // add_query_arg() return the current url

                $add_query_args = array(
                    "status" => 'duplicated'
                );
                if( isset( $_GET['fstatus'] ) && $_GET['fstatus'] != '' ){
                    $add_query_args['fstatus'] = sanitize_text_field( $_GET['fstatus'] );
                }
                $url = remove_query_arg( array('action', 'id', '_wpnonce') );
                $url = esc_url_raw( add_query_arg( $add_query_args, $url ) );
                wp_redirect( $url );
            }

        }

        // If the delete bulk action is triggered
        if ( ( isset( $_POST['action'] ) && $_POST['action'] == 'bulk-delete' ) || ( isset( $_POST['action2'] ) && $_POST['action2'] == 'bulk-delete' ) ) {

            $delete_ids = ( isset( $_POST['bulk-delete'] ) && ! empty( $_POST['bulk-delete'] ) ) ? esc_sql( $_POST['bulk-delete'] ) : array();

            // loop over the array of record IDs and delete them
            foreach ( $delete_ids as $id ) {
                self::delete_items( $id );
            }

            // esc_url_raw() is used to prevent converting ampersand in url to "#038;"
            // add_query_arg() return the current url
            $add_query_args = array(
                "status" => 'all-deleted'
            );
            if( isset( $_GET['fstatus'] ) && $_GET['fstatus'] != '' ){
                $add_query_args['fstatus'] = sanitize_text_field( $_GET['fstatus'] );
            }
            $url = remove_query_arg( array('action', 'id', '_wpnonce') );
            $url = esc_url_raw( add_query_arg( $add_query_args, $url ) );
            wp_redirect( $url );
        }

        // If the delete bulk action is triggered
        if ( ( isset( $_POST['action'] ) && $_POST['action'] == 'bulk-trash' ) || ( isset( $_POST['action2'] ) && $_POST['action2'] == 'bulk-trash' ) ) {

            $trash_ids = ( isset( $_POST['bulk-delete'] ) && ! empty( $_POST['bulk-delete'] ) ) ? esc_sql( $_POST['bulk-delete'] ) : array();

            // loop over the array of record IDs and delete them
            foreach ( $trash_ids as $id ) {
                self::trash_items( $id );
            }

            // esc_url_raw() is used to prevent converting ampersand in url to "#038;"
            // add_query_arg() return the current url
            $add_query_args = array(
                "status" => 'all-trashed'
            );
            if( isset( $_GET['fstatus'] ) && $_GET['fstatus'] != '' ){
                $add_query_args['fstatus'] = sanitize_text_field( $_GET['fstatus'] );
            }
            $url = remove_query_arg( array('action', 'id', '_wpnonce') );
            $url = esc_url_raw( add_query_arg( $add_query_args, $url ) );
            wp_redirect( $url );
        }

        // If the delete bulk action is triggered
        if ( ( isset( $_POST['action'] ) && $_POST['action'] == 'bulk-restore' ) || ( isset( $_POST['action2'] ) && $_POST['action2'] == 'bulk-restore' ) ) {

            $restore_ids = ( isset( $_POST['bulk-delete'] ) && ! empty( $_POST['bulk-delete'] ) ) ? esc_sql( $_POST['bulk-delete'] ) : array();

            // loop over the array of record IDs and delete them
            foreach ( $restore_ids as $id ) {
                self::restore_items( $id );
            }

            // esc_url_raw() is used to prevent converting ampersand in url to "#038;"
            // add_query_arg() return the current url
            $add_query_args = array(
                "status" => 'all-restored'
            );
            if( isset( $_GET['fstatus'] ) && $_GET['fstatus'] != '' ){
                $add_query_args['fstatus'] = sanitize_text_field( $_GET['fstatus'] );
            }
            $url = remove_query_arg( array('action', 'id', '_wpnonce') );
            $url = esc_url_raw( add_query_arg( $add_query_args, $url ) );
            wp_redirect( $url );
        }

        // If the delete bulk action is triggered
        if ( ( isset( $_POST['action'] ) && $_POST['action'] == 'bulk-duplicate' ) || ( isset( $_POST['action2'] ) && $_POST['action2'] == 'bulk-duplicate' ) ) {

            $restore_ids = ( isset( $_POST['bulk-delete'] ) && ! empty( $_POST['bulk-delete'] ) ) ? esc_sql( $_POST['bulk-delete'] ) : array();

            // loop over the array of record IDs and delete them
            foreach ( $restore_ids as $id ) {
                $this->duplicate_items( $id );
            }

            // esc_url_raw() is used to prevent converting ampersand in url to "#038;"
            // add_query_arg() return the current url
            $add_query_args = array(
                "status" => 'all-duplicated'
            );
            if( isset( $_GET['fstatus'] ) && $_GET['fstatus'] != '' ){
                $add_query_args['fstatus'] = sanitize_text_field( $_GET['fstatus'] );
            }
            $url = remove_query_arg( array('action', 'id', '_wpnonce') );
            $url = esc_url_raw( add_query_arg( $add_query_args, $url ) );
            wp_redirect( $url );
        }
    }

    public function sf_forms_notices(){
        $status = (isset($_REQUEST['status'])) ? sanitize_text_field( $_REQUEST['status'] ) : '';

        if ( empty( $status ) )
            return;

        $error = false;
        switch ( $status ) {
            case 'created':
                $updated_message = esc_html( __( 'Form created.', $this->plugin_name ) );
                break;
            case 'updated':
                $updated_message = esc_html( __( 'Form saved.', $this->plugin_name ) );
                break;
            case 'duplicated':
                $updated_message = esc_html( __( 'Form duplicated.', $this->plugin_name ) );
                break;
            case 'deleted':
                $updated_message = esc_html( __( 'Form deleted.', $this->plugin_name ) );
                break;
            case 'trashed':
                $updated_message = esc_html( __( 'Form moved to trash.', $this->plugin_name ) );
                break;
            case 'restored':
                $updated_message = esc_html( __( 'Form restored.', $this->plugin_name ) );
                break;
            case 'all-duplicated':
                $updated_message = esc_html( __( 'Forms are duplicated.', $this->plugin_name ) );
                break;
            case 'all-deleted':
                $updated_message = esc_html( __( 'Forms are deleted.', $this->plugin_name ) );
                break;
            case 'all-trashed':
                $updated_message = esc_html( __( 'Forms are moved to trash.', $this->plugin_name ) );
                break;
            case 'all-restored':
                $updated_message = esc_html( __( 'Forms are restored.', $this->plugin_name ) );
                break;
            case 'empty-title':
                $error = true;
                $updated_message = esc_html( __( 'Error: Form title can not be empty.', $this->plugin_name ) );
                break;
            default:
                break;
        }

        if ( empty( $updated_message ) )
            return;

        $notice_class = 'success';
        if( $error ){
            $notice_class = 'error';
        }
        ?>
        <div class="notice notice-<?php echo esc_attr( $notice_class ); ?> is-dismissible">
            <p> <?php echo $updated_message; ?> </p>
        </div>
        <?php
    }
}
