<?php
/**
 * Plugin Name: UpStroke: Dynamic Shipping
 * Plugin URI: https://buildwoofunnels.com
 * Description: This addon calculates shipping cost of Upsell items based on shipping method selected in primary order
 * Version: 1.6.0
 * Author: WooFunnels
 * Author URI: https://buildwoofunnels.com
 * Text Domain: woofunnels-upstroke-dynamic-shipping
 * Domain Path: /languages/
 *
 * Requires at least: 4.9.0
 * Tested up to: 5.2.3
 * WC requires at least: 3.3.1
 * WC tested up to: 3.5.0
 * WooFunnels: true
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! function_exists( 'woofunnels_upstroke_dependency' ) ) {

	/**
	 * Function to check if woofunnels upstroke pro version is loaded and activated or not?
	 * @return bool True|False
	 */
	function woofunnels_upstroke_dependency() {

		$active_plugins = (array) get_option( 'active_plugins', array() );

		if ( is_multisite() ) {
			$active_plugins = array_merge( $active_plugins, get_site_option( 'active_sitewide_plugins', array() ) );
		}

		if ( false === file_exists( plugin_dir_path( __DIR__ ) . 'woofunnels-upstroke-one-click-upsell/woofunnels-upstroke-one-click-upsell.php' ) ) {
			return false;
		}

		return in_array( 'woofunnels-upstroke-one-click-upsell/woofunnels-upstroke-one-click-upsell.php', $active_plugins ) || array_key_exists( 'woofunnels-upstroke-one-click-upsell/woofunnels-upstroke-one-click-upsell.php', $active_plugins );
	}
}

class WooFunnels_UpStroke_Dynamic_Shipping {

	public static $instance;

	public function __construct() {

		$this->init_constants();
		$this->init_hooks();
	}

	public function init_constants() {
		define( 'WF_UPSTROKE_DP_VERSION', '1.6.0' );
		define( 'WFOCU_MIN_DP_VERSION', '1.11.5' );
		define( 'WF_UPSTROKE_DP_BASENAME', plugin_basename( __FILE__ ) );
	}

	public static function instance() {
		if ( null == self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function init_hooks() {
		add_action( 'admin_enqueue_scripts', array( $this, 'maybe_render_assets' ) );
		add_action( 'plugins_loaded', array( $this, 'maybe_check_version' ) );

		/**
		 * API receiving hook to catch the paypal response and process billing agreement creation
		 */
		add_action( 'woocommerce_api_wfocu_cs', array( $this, 'maybe_handle_call_cs' ) );
		add_action( 'plugins_loaded', array( $this, 'add_licence_support_file' ) );

		add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );
	}

	public function maybe_render_assets() {

		if ( false === class_exists( 'WFOCU_Common' ) ) {
			return;
		}
		if ( true === WFOCU_Common::is_load_admin_assets( 'builder' ) ) {

			wp_enqueue_script( 'wfocu_dynamic_shipping_script', plugin_dir_url( __FILE__ ) . 'assets/js/admin.js', array( 'wfocu-admin-builder' ) );

		}
	}

	public function add_licence_support_file() {
		include_once plugin_dir_path( __FILE__ ) . 'class-woofunnels-support-dynamic-shipping.php';
	}

	public function calculate_dynamic_shipping( $products, $location, $existing_methods, $order ) {
		$get_free_shipping  = array();
		$shipping           = array();
		$get_shipping_items = $order->get_items( 'shipping' );

		if ( ! empty( $products ) ) {
			$response = wp_remote_post( WC()->api_request_url( 'wfocu_cs' ), array(
				'body'      => array(
					'products'        => $products,
					'location'        => $location,
					'chosen_shipping' => $existing_methods,
				),
				'sslverify' => false,
				'timeout'   => 20,
			) );

			if ( is_wp_error( $response ) ) {
				//handle error
			} else {

				$response_packages = json_decode( wp_remote_retrieve_body( $response ), true );
				$order_behavior    = WFOCU_Core()->funnels->get_funnel_option( 'order_behavior' );
				$is_batching_on    = ( 'batching' === $order_behavior ) ? true : false;
				$override          = false;
				$shipping_prev     = array();

				WFOCU_Core()->log->log( 'WFOCU Existing methods: ' . print_r( $existing_methods, true ) );
				WFOCU_Core()->log->log( 'WFOCU Response packages: ' . print_r( $response_packages, true ) );

				if ( isset( $response_packages['packages'] ) && is_array( $response_packages['packages'] ) && is_array( $response ) && count( $response ) > 0 ) {
					foreach ( $response_packages['packages'] as $package ) {
						foreach ( $package as $method_id => $method_data ) {
							/**
							 * Few times we need all the shipping options to render in the offer
							 * In this case we check if parent method is not provided then return All the methods
							 */
							if ( count( $existing_methods ) > 0 && in_array( $method_id, $existing_methods, true ) && $is_batching_on ) {
								$shipping   = array();
								$shipping[] = array(
									$method_id => $method_data,
								);
								$override   = false;
								foreach ( is_array( $get_shipping_items ) ? $get_shipping_items : array() as $shipping_item ) {
									if ( $method_id === $shipping_item->get_method_id() ) {
										$shipping_prev = array(
											'cost' => $shipping_item->get_total(),
											'tax'  => $shipping_item->get_total_tax(),
										);
									}
								}
								break;
							} else {
								$shipping[] = array(
									$method_id => $method_data,
								);
								$override   = true;
							}
							/*if ( count( $existing_methods ) === 0 ) { //When primary order is virtual thus no shipping
								$shipping[] = array(
									$method_id => $method_data,
								);
							}*/
						}
					}

					//Iteration for free shipping
					foreach ( $response_packages['packages'] as $package ) {
						foreach ( $package as $method_id => $method_data ) {
							if ( WFOCU_Core()->shipping->is_free_shipping( $method_data['method'] ) && count( $get_free_shipping ) === 0 ) {
								$get_free_shipping = array(
									$method_id => $method_data,
								);
							}
						}
					}
				}
			}
		}

		if ( count( $shipping_prev ) > 0 ) {
			return array(
				'free_shipping' => $get_free_shipping,
				'shipping'      => $shipping,
				'override'      => $override,
				'shipping_prev' => $shipping_prev,
			);
		} else {
			return array(
				'free_shipping' => $get_free_shipping,
				'shipping'      => $shipping,
				'override'      => $override,
			);
		}

	}

	/**
	 * @hooked over `maybe_handle_call_cs`
	 * This methods creates an environment and sets up the cart to find the shipping cost dynamically
	 * 1. Adds products to the cart
	 * 2. Sets up users location so that correct taxes and shipping costs get calculated
	 * 3. Calculates shipping & shipping taxes
	 * Sends a json response of shipping packages
	 * @throws Exception
	 * @throws WC_Data_Exception
	 */
	public function maybe_handle_call_cs() {
		/**
		 * @todo optimization needed at this place
		 */
		ob_start();
		list( $country, $state, $city, $postcode ) = $_POST['location'];
		WC()->customer->set_location( $country, $state, $postcode, $city );
		WC()->customer->set_shipping_location( $country, $state, $postcode, $city );

		$products = $_POST['products'];
		WC()->session->set( 'chosen_shipping_methods', $_POST['chosen_shipping'] );
		WFOCU_Core()->log->log( 'processed products: ' . print_r( $products, true ) );

		foreach ( $products as $product ) {
			WC()->session->set( 'chosen_shipping_methods', $_POST['chosen_shipping'] );

			try {
				$offer_product = ( isset( $product['offer_product'] ) && ( $product['offer_product'] ) ) ? array(
					'offer_product' => 1,
				) : array();
				$cart_item_id  = WC()->cart->add_to_cart( $product['product_id'], $product['qty'], $product['variation_id'], $offer_product );

			} catch ( Exception $e ) {
				WFOCU_Core()->log->log( 'reason no add to cart: ' . print_r( $e, true ) );
			}

			if ( ! isset( $cart_item_id ) && is_null( $cart_item_id ) ) {
				continue;
			}

			if ( ! is_object( WC()->cart->get_cart_contents()[ $cart_item_id ]['data'] ) ) {
				continue;
			}
			WFOCU_Core()->log->log( 'setting up price: ' . print_r( $product['price'], true ) );

			/**
			 * It is important to set the price of the item just added to the cart,
			 * because if we do not set the correct price there might be different cart totals for the current scenarios as woocommerce will pick default pricing of a product
			 * so suppose if a product is of $10 in real and in upsell its of $8 then cart totals should have $8 not $10
			 * If order total mismatches with the total showing user to the cart then, there might be free shipping offered right away even when offer total doesn't reach threshold
			 */
			WC()->cart->get_cart_contents()[ $cart_item_id ]['data']->set_price( $product['price'] );
		}

		WC()->session->set( 'chosen_shipping_methods', $_POST['chosen_shipping'] );
		WC()->cart->calculate_shipping();
		WC()->cart->calculate_totals();

		$packages      = WC()->shipping->get_packages();
		$offer_package = array();

		foreach ( is_array( $packages ) ? $packages : array() as $key => $package ) {
			if ( count( $offer_package ) > 0 ) {
				break;
			}
			if ( isset( $package['contents'] ) ) {
				foreach ( $package['contents'] as $pkey => $pval ) {
					if ( isset( $pval['variation'] ) && isset( $pval['variation']['offer_product'] ) && 1 === $pval['variation']['offer_product'] ) {
						$offer_package[] = $packages[ $key ];
						break;
					}
				}
			}
		}

		$my_packages = array();
		foreach ( count( $offer_package ) > 0 ? $offer_package : $packages as $i => $package ) {
			$my_packages[ $i ] = $this->wfocu_parse_shipping_packages( $package['rates'] );
		}
		if ( defined( 'WFOCU_IS_DEV' ) && true === WFOCU_IS_DEV ) {
			WFOCU_Core()->log->log( 'Final parsed offer shipping Packages: ' . print_r( $my_packages, true ) );
		}

		WC()->cart->empty_cart();
		WC()->session->destroy_session();

		ob_get_clean();
		wp_send_json( array(
			'packages' => $my_packages,
		) );
	}

	public function wfocu_parse_shipping_packages( $package ) {
		$custom_package = array();
		if ( $package && is_array( $package ) && count( $package ) > 0 ) {
			foreach ( $package as $key => $ship ) {
				$custom_package[ $key ] = array(
					'method'       => $ship->get_method_id(),
					'label'        => $ship->get_label(),
					'cost'         => $ship->get_cost(),
					'shipping_tax' => $ship->get_shipping_tax(),
					'taxes'        => $ship->get_taxes(),

				);
			}
		}

		return $custom_package;
	}

	public function wfocu_version_check_notice() {
		?>
        <div class="error">
            <p>
				<?php
				/* translators: %1$s: Min required woocommerce version */
				printf( __( '<strong> Attention: </strong>UpStroke Dynamic Shipping requires  WooFunnels UpStroke: One Click Upsell version %1$s or greater. Kindly update the WooFunnels UpStroke: One Click Upsell plugin.', 'woofunnels-upstroke-dynamic-shipping' ), WFOCU_MIN_DP_VERSION );
				?>
            </p>
        </div>
		<?php
	}

	public function maybe_check_version() {
		if ( ! version_compare( WFOCU_VERSION, WFOCU_MIN_DP_VERSION, '>=' ) ) {
			add_action( 'admin_notices', array( $this, 'wfocu_version_check_notice' ) );

			return false;
		}
	}

	public function load_textdomain() {

		$locale = is_admin() && function_exists( 'get_user_locale' ) ? get_user_locale() : get_locale();

		unload_textdomain( 'woofunnels-upstroke-dynamic-shipping' );
		load_textdomain( 'woofunnels-upstroke-dynamic-shipping', WP_LANG_DIR . '/woofunnels-upstroke-dynamic-shipping-' . $locale . '.mo' );

		load_plugin_textdomain( 'woofunnels-upstroke-dynamic-shipping', false, plugin_basename( dirname( __FILE__ ) ) . '/languages' );


	}
}

if ( true === woofunnels_upstroke_dependency() ) {
	WooFunnels_UpStroke_Dynamic_Shipping::instance();

}