<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WFOCU_Gateway_Integration_WooCommerce_Payments class.
 *
 * @extends WFOCU_Gateway
 */
class WFOCU_Gateway_Integration_WooCommerce_Payments extends WFOCU_Gateway {


	protected static $ins = null;
	public $key = 'woocommerce_payments';
	public $token = false;
	public $has_intent_secret = false;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->refund_supported = true;

		parent::__construct();
		add_filter( 'wc_payments_display_save_payment_method_checkbox', array( $this, 'should_tokenize_gateway' ) );
		add_action( 'wfocu_front_pre_init_funnel_hooks', array( $this, 'maybe_force_save_token_for_3ds' ), 1 );
		add_action( 'wfocu_footer_before_print_scripts', array( $this, 'maybe_render_in_offer_transaction_scripts' ), 999 );
		add_filter( 'wfocu_allow_ajax_actions_for_charge_setup', array( $this, 'allow_check_action' ) );
		add_action( 'wp_footer', array( $this, 'maybe_render_script_to_allow_tokenization' ) );
	}

	public static function get_instance() {
		if ( null === self::$ins ) {
			self::$ins = new self;
		}

		return self::$ins;
	}


	public function should_tokenize_gateway( $display_method ) {

		if ( false !== $this->is_enabled() ) {
			return false;
		}


		return $display_method;
	}

	/**
	 * Try and get the payment token saved by the gateway
	 *
	 * @param WC_Order $order
	 *
	 * @return true|false on success false otherwise
	 */
	public function has_token( $order ) {
		$this->token = $this->get_payment_token( $order );


		if ( ! empty( $this->token ) ) {
			return true;
		}

		return false;

	}


	/**
	 * Try and get the payment token saved by the gateway
	 *
	 * @param WC_Order $order
	 *
	 * @return WC_Payment_Token|false on success false otherwise
	 */
	public function get_token( $order ) {
		$this->token = $this->get_payment_token( $order );


		if ( ! empty( $this->token ) ) {
			return $this->token;
		}

		return false;

	}

	/**
	 * Retrieve payment token from an ordeer
	 *
	 * @param WC_Order $order Order
	 *
	 * @return null|WC_Payment_Token Last token associated with order
	 */
	protected function get_payment_token( $order ) {
		$order_tokens = $order->get_payment_tokens();
		$token_id     = end( $order_tokens );

		return ! $token_id ? null : WC_Payment_Tokens::get( $token_id );
	}


	/**
	 * Model method to handle the client payments aka In-offer transactions
	 * This primary covers both operations 1) init of client payment
	 * 2) auth of client operations
	 * Also handles further API operation to mark success and failtures
	 */
	public function process_client_payment() {

		/**
		 * Prepare and populate client collected data to process further.
		 */
		$get_current_offer      = WFOCU_Core()->data->get( 'current_offer' );
		$get_current_offer_meta = WFOCU_Core()->offers->get_offer_meta( $get_current_offer );
		WFOCU_Core()->data->set( '_offer_result', true );
		$posted_data = WFOCU_Core()->process_offer->parse_posted_data( $_POST ); // phpcs:ignore WordPress.Security.NonceVerification.Missing

		/**
		 * return if found error in the charge request
		 */
		if ( false === WFOCU_AJAX_Controller::validate_charge_request( $posted_data ) ) {
			wp_send_json( array(
				'result' => 'error',
			) );
		}


		/**
		 * Setup the upsell to initiate the charge process
		 */
		WFOCU_Core()->process_offer->execute( $get_current_offer_meta );

		$get_order = WFOCU_Core()->data->get_parent_order();


		$intent_from_posted = filter_input( INPUT_POST, 'intent', FILTER_SANITIZE_NUMBER_INT );

		/**
		 * If intent flag set found in the posted data from the client then it means we just need to verify the intent status and then process failure or success
		 * if not found it means that its the first initial intent creation call
		 */
		if ( ! empty( $intent_from_posted ) ) {


			/**
			 * process response when user either failed or approve the auth.
			 */
			$intent_secret_from_posted = filter_input( INPUT_POST, 'intent_secret', FILTER_UNSAFE_RAW );

			/**
			 * If not found the intent secret with the flag then fail, there could be few security issues
			 */
			if ( empty( $intent_secret_from_posted ) ) {
				$this->handle_api_error( esc_attr__( 'Offer payment failed. Reason: Intent secret missing from auth', 'woofunnels-upstroke-one-click-upsell' ), 'Intent secret missing from auth', $get_order, true );
			}

			/**
			 * get intent ID from the data session
			 */
			$get_intent_id_from_posted_secret = WFOCU_Core()->data->get( 'c_intent_secret_' . $intent_secret_from_posted, '', 'gateway' );
			if ( empty( $get_intent_id_from_posted_secret ) ) {
				$this->handle_api_error( esc_attr__( 'Offer payment failed. Reason: Unable to find matching ID for the secret', 'woofunnels-upstroke-one-click-upsell' ), 'Unable to find matching ID for the secret', $get_order, true );
			}

			/**
			 * Get intent from the payment gateway API
			 */
			$intent = WC_Payments::get_payments_api_client()->get_intent( $get_intent_id_from_posted_secret );


			if ( false !== $intent ) {
				WFOCU_Core()->data->set( '_transaction_id', $get_intent_id_from_posted_secret );
				wp_send_json( array(
					'result'   => 'success',
					'response' => WFOCU_Core()->process_offer->_handle_upsell_charge( true ),
				) );
			}

		} else {


			/**
			 * get token from the order and try to create and verify intent
			 *
			 */
			try {
				$get_package = WFOCU_Core()->data->get( '_upsell_package' );
				$token       = $this->get_token( $get_order );

				$payment_information = new WCPay\Payment_Information( '', $get_order, WCPay\Constants\Payment_Type::SINGLE(), $token, WCPay\Constants\Payment_Initiated_By::CUSTOMER() );
				$response            = $this->process_payment_for_upsell( $get_package, $payment_information );

			} catch ( Exception $e ) {
				/**
				 * If error captured during charge process, then handle as failure
				 */
				$this->handle_api_error( esc_attr__( 'Offer payment failed. Reason: ' . $e->getMessage() . '', 'woofunnels-upstroke-one-click-upsell' ), 'Error Captured: ' . print_r( $e->getMessage() . " <-- Generated on" . $e->getFile() . ":" . $e->getLine(), true ), $get_order, true ); // @codingStandardsIgnoreLine

			}
			/**
			 * check if response status
			 */
			if ( 'on-hold' === $response['status'] ) {
				/**
				 * on hold status shows that intent was created and we have intent to further requires user action
				 * Save the intent_secret in the session and return appropiate response
				 */
				if ( isset( $response['data']['intent_secret'] ) ) {
					WFOCU_Core()->data->set( 'c_intent_secret_' . $response['data']['intent_secret'], $response['data']['id'], 'gateway' );
					WFOCU_Core()->data->save( 'gateway' );
					wp_send_json( array(
						'result'   => 'success',
						'response' => $response,
					) );
				}
			} elseif ( 'success' === $response['status'] ) {
				WFOCU_Core()->data->set( '_transaction_id', $response['transaction_id'] );

				/**
				 * Handle success
				 */
				$data = WFOCU_Core()->process_offer->_handle_upsell_charge( true );
			} else {
				/**
				 * Handle failture
				 */
				$data = WFOCU_Core()->process_offer->_handle_upsell_charge( false );
			}

		}


		wp_send_json( array(
			'result'   => 'success',
			'response' => $data,
		) );
	}

	/**
	 * This function is placed here as a fallback function when JS client side integration fails mysteriosly
	 * It creates intent and then try to confirm that intent, if successfull then mark success, otherwise failure
	 *
	 * @param WC_Order $order
	 *
	 * @return true
	 * @throws WFOCU_Payment_Gateway_Exception
	 */
	public function process_charge( $order ) {
		$is_successful = false;


		$token       = $this->get_token( $order );
		$get_package = WFOCU_Core()->data->get( '_upsell_package' );

		$payment_information = new WCPay\Payment_Information( '', $order, WCPay\Constants\Payment_Type::SINGLE(), $token, WCPay\Constants\Payment_Initiated_By::CUSTOMER() );
		$response            = $this->process_payment_for_upsell( $get_package, $payment_information );


		if ( $response['status'] !== 'success' ) {


			$is_successful = false;
			throw new WFOCU_Payment_Gateway_Exception( "Stripe : " . $response['data']['error'], 102, $this->get_key() );

		} else {
			WFOCU_Core()->data->set( '_transaction_id', $response['transaction_id'] );
			$is_successful = true;

		}


		return $this->handle_result( $is_successful );
	}

	public function process_payment_for_upsell( $package, $payment_information ) {
		$order               = $payment_information->get_order();


		$amount   = $package['total'];
		$name     = sanitize_text_field( $order->get_billing_first_name() ) . ' ' . sanitize_text_field( $order->get_billing_last_name() );
		$email    = sanitize_email( $order->get_billing_email() );
		$metadata = [
			'customer_name'  => $name,
			'customer_email' => $email,
			'site_url'       => esc_url( get_site_url() ),
			'payment_type'   => $payment_information->get_payment_type(),
		];


		$customer_id = $order->get_meta( '_stripe_customer_id' );

		$plugins = get_plugins();


        if(version_compare('3.9.0', $plugins['woocommerce-payments/woocommerce-payments.php']['Version'],'<=')) {
	        // Create intention, try to confirm it & capture the charge (if 3DS is not required).
	        $intent = WC_Payments::get_payments_api_client()->create_and_confirm_intention(
		        WC_Payments_Utils::prepare_amount( $amount, $order->get_currency() ),
		        strtolower( $order->get_currency() ),
		        $payment_information->get_payment_method(),
		        $customer_id,
		        $payment_information->is_using_manual_capture(),
		        $payment_information->should_save_payment_method_to_store(),
		        $payment_information->should_save_payment_method_to_platform(),
		        $metadata
	        );

        }else{
	        $intent = WC_Payments::get_payments_api_client()->create_and_confirm_intention( WC_Payments_Utils::prepare_amount( $amount, $order->get_currency() ), strtolower( $order->get_currency() ), $payment_information->get_payment_method(), $customer_id, $payment_information->is_using_manual_capture(), $payment_information->should_save_payment_method(), $metadata, [], $payment_information->is_merchant_initiated() );


        }

		$intent_id     = $intent->get_id();
		$status        = $intent->get_status();
		$client_secret = $intent->get_client_secret();
		$next_action   = $intent->get_next_action();


		if ( ! empty( $intent ) ) {


			if ( 'requires_action' === $status ) {
				if ( isset( $next_action['type'] ) && 'redirect_to_url' === $next_action['type'] && ! empty( $next_action['redirect_to_url']['url'] ) ) {
					//@todo deal this scenario later
				} else {
					/**
					 * return intent_secret as the data to the client so that necesary next operations could taken care.
					 */

					return array( 'status' => 'on-hold', 'transaction_id' => '', 'data' => [ 'intent_secret' => $client_secret, 'id' => $intent_id ] );
				}
			}
			if ( ! in_array( $status, [ 'succeeded', 'requires_capture', 'processing' ], true ) ) {
				return array( 'status' => 'failture', 'transaction_id' => $intent_id, 'data' => [ 'error' ] );
			}
		}

		return array( 'status' => 'success', 'transaction_id' => $intent_id, 'data' => [] );

	}


	/**
	 * Handling refund offer request from admin
	 *
	 * @throws WC_Stripe_Exception
	 */
	public function process_refund_offer( $order ) { //phpcs:ignore VariableAnalysis.CodeAnalysis.VariableAnalysis.UnusedParameter
		$refund_data = $_POST;  // phpcs:ignore WordPress.Security.NonceVerification.Missing

		$txn_id = isset( $refund_data['txn_id'] ) ? $refund_data['txn_id'] : '';
		$amnt   = isset( $refund_data['amt'] ) ? $refund_data['amt'] : '';

		if ( ! is_null( $amnt ) ) {


			try {
				/**
				 * Get intent object from the saved transaction ID (payment intent ID)
				 */
				$intent = WC_Payments::get_payments_api_client()->get_intent( $txn_id );

				/**
				 * Perform refund API call
				 */
				WC_Payments::get_payments_api_client()->refund_charge( $intent->get_charge_id(), WC_Payments_Utils::prepare_amount( $amnt, $order->get_currency() ) );

			} catch ( Exception $e ) {

				WFOCu_Core()->log->log( 'Offer refund failed, reason' . $e->getMessage() );

				return false;
			}


		}

	}


	/**
	 * Render Javascript that is responsible for client side payment
	 */
	public function maybe_render_in_offer_transaction_scripts() {
		$order = WFOCU_Core()->data->get_current_order();

		if ( ! $order instanceof WC_Order ) {
			return;
		}

		if ( $this->get_key() !== $order->get_payment_method() ) {
			return;
		}
		$all_js_config = $this->get_wc_gateway()->get_payment_fields_js_config();
		?>
		<script type="text/javascript" src="https://js.stripe.com/v3/?ver=3.0"></script> <?php //phpcs:ignore WordPress.WP.EnqueuedResources.NonEnqueuedScript ?>

		<script type="text/javascript">

            (
                function ($) {
                    "use strict";
                    var wfocuWCPAY = Stripe('<?php echo esc_js( $all_js_config["publishableKey"] ); ?>', {
                        stripeAccount: '<?php echo esc_js( $all_js_config["accountId"] ); ?>',
                        locale: '<?php echo esc_js( $all_js_config["locale"] ); ?>'
                    });

                    var wfocuWCPAYJS = {
                        bucket: null,
                        initCharge: function () {
                            var getBucketData = this.bucket.getBucketSendData();

                            var postData = $.extend(getBucketData, {action: 'wfocu_front_handle_wcpay_payments'});

                            var action = $.post(wfocu_vars.wc_ajax_url.toString().replace('%%endpoint%%', 'wfocu_front_handle_wcpay_payments'), postData);

                            action.done(function (data) {

                                /**
                                 * Process the response for the call to handle client stripe payments
                                 * first handle error state to show failure notice and redirect to thank you
                                 * */
                                if (data.result !== "success") {

                                    wfocuWCPAYJS.bucket.swal.show({'text': wfocu_vars.messages.offer_msg_pop_failure, 'type': 'warning'});
                                    if (typeof data.response !== "undefined" && typeof data.response.redirect_url !== 'undefined') {

                                        setTimeout(function () {
                                            window.location = data.response.redirect_url;
                                        }, 1500);
                                    } else {
                                        /** move to order received page */
                                        if (typeof wfocu_vars.order_received_url !== 'undefined') {

                                            window.location = wfocu_vars.order_received_url + '&ec=wc_api1';

                                        }
                                    }
                                } else {
                                    /**
                                     * There could be two states --
                                     * 1. intent confirmed
                                     * 2. requires action
                                     * */

                                    /**
                                     * handle scenario when authentication requires for the payment intent
                                     * In this case we need to trigger stripe payment intent popups
                                     * */
                                    if (typeof data.response.data !== "undefined" && typeof data.response.data.intent_secret !== "undefined" && '' !== data.response.data.intent_secret) {

                                        wfocuWCPAY.confirmCardPayment(data.response.data.intent_secret)
                                            .then(function (response) {
                                                if (response.error) {
                                                    throw response.error;
                                                }

                                                if ('requires_capture' !== response.paymentIntent.status && 'succeeded' !== response.paymentIntent.status) {
                                                    return;
                                                }
                                                $(document).trigger('wfocuWCPAYOnAuthentication', [response, true]);
                                                return;

                                            })
                                            .catch(function (error) {
                                                $(document).trigger('wfocuWCPAYOnAuthentication', [false, false]);
                                                return;

                                            });
                                        return;
                                    }
                                    /**
                                     * If code reaches here means it no longer require any authentication from the client and we process success
                                     * */

                                    wfocuWCPAYJS.bucket.swal.show({'text': wfocu_vars.messages.offer_success_message_pop, 'type': 'success'});
                                    if (typeof data.response !== "undefined" && typeof data.response.redirect_url !== 'undefined') {

                                        setTimeout(function () {
                                            window.location = data.response.redirect_url;
                                        }, 1500);
                                    } else {
                                        /** move to order received page */
                                        if (typeof wfocu_vars.order_received_url !== 'undefined') {

                                            window.location = wfocu_vars.order_received_url + '&ec=wc_api4';

                                        }
                                    }
                                }
                            });
                            action.fail(function (data) {

                                /**
                                 * In case of failure of ajax, process failure
                                 * */
                                wfocuWCPAYJS.bucket.swal.show({'text': wfocu_vars.messages.offer_msg_pop_failure, 'type': 'warning'});
                                if (typeof data.response !== "undefined" && typeof data.response.redirect_url !== 'undefined') {

                                    setTimeout(function () {
                                        window.location = data.response.redirect_url;
                                    }, 1500);
                                } else {
                                    /** move to order received page */
                                    if (typeof wfocu_vars.order_received_url !== 'undefined') {

                                        window.location = wfocu_vars.order_received_url + '&ec=wc_api3';

                                    }
                                }
                            });
                        }
                    }

                    /**
                     * Handle popup authentication results
                     */
                    $(document).on('wfocuWCPAYOnAuthentication', function (e, response, is_success) {

                        if (is_success) {
                            var postData = $.extend(wfocuWCPAYJS.bucket.getBucketSendData(), {
                                action: 'wfocu_front_handle_wcpay_payments',
                                intent: 1,
                                intent_secret: response.paymentIntent.client_secret
                            });

                        } else {
                            var postData = $.extend(wfocuWCPAYJS.bucket.getBucketSendData(), {action: 'wfocu_front_handle_wcpay_payments', intent: 1, intent_secret: ''});

                        }
                        var action = $.post(wfocu_vars.wc_ajax_url.toString().replace('%%endpoint%%', 'wfocu_front_handle_wcpay_payments'), postData);
                        action.done(function (data) {
                            if (data.result !== "success") {
                                wfocuWCPAYJS.bucket.swal.show({'text': wfocu_vars.messages.offer_msg_pop_failure, 'type': 'warning'});
                            } else {
                                wfocuWCPAYJS.bucket.swal.show({'text': wfocu_vars.messages.offer_success_message_pop, 'type': 'success'});
                            }
                            if (typeof data.response !== "undefined" && typeof data.response.redirect_url !== 'undefined') {

                                setTimeout(function () {
                                    window.location = data.response.redirect_url;
                                }, 1500);
                            } else {
                                /** move to order received page */
                                if (typeof wfocu_vars.order_received_url !== 'undefined') {

                                    window.location = wfocu_vars.order_received_url + '&ec=stripe_error2';

                                }
                            }
                        });
                    });

                    /**
                     * Save the bucket instance at several
                     */
                    $(document).on('wfocuBucketCreated', function (e, Bucket) {
                        wfocuWCPAYJS.bucket = Bucket;

                    });
                    $(document).on('wfocu_external', function (e, Bucket) {
                        /**
                         * Check if we need to mark inoffer transaction to prevent default behavior of page
                         */
                        if (0 !== Bucket.getTotal()) {
                            Bucket.inOfferTransaction = true;
                            wfocuWCPAYJS.initCharge();
                        }
                    });

                    $(document).on('wfocuBucketConfirmationRendered', function (e, Bucket) {
                        wfocuWCPAYJS.bucket = Bucket;

                    });
                    $(document).on('wfocuBucketLinksConverted', function (e, Bucket) {
                        wfocuWCPAYJS.bucket = Bucket;

                    });
                })(jQuery);
		</script>
		<?php
	}

	/**
	 * maybe save token in case of 3ds flow
	 *
	 * @param WC_Order $order
	 */
	public function maybe_force_save_token_for_3ds( $order ) {

		/**
		 * Check if we have the correct ajax process to work onto
		 */
		if ( did_action( 'wp_ajax_nopriv_update_order_status' ) || did_action( 'wp_ajax_update_order_status' ) ) {
			try {
				$payment_method_id = isset( $_POST['payment_method_id'] ) ? wc_clean( wp_unslash( $_POST['payment_method_id'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Missing

				$customer_service = new WC_Payments_Customer_Service( WC_Payments::get_payments_api_client(), WC_Payments::get_account_service() );
				$token_service    = new WC_Payments_Token_Service( WC_Payments::get_payments_api_client(), $customer_service );

				/**
				 * tell gateway to save the payment method (token) to the user
				 */
				$token = $token_service->add_payment_method_to_user( $payment_method_id, wp_get_current_user() );
				$this->get_wc_gateway()->add_token_to_order( $order, $token );

				/**
				 * force readmeta data to avoid any object caching scenarios
				 */
				$order->read_meta_data( true );
			} catch ( Exception $e ) {
				// If saving the token fails, log the error message but catch the error to avoid crashing the checkout flow
				WFOCU_Core()->log->log( $order->get_id() . ' Unable to save early token for 3ds scenarios' . $e->getMessage() );
			}
		}
	}

	/**
	 * Allow action of wcpayments ajax
	 *
	 * @param array $actions
	 *
	 * @return mixed modified actions
	 */
	public function allow_check_action( $actions ) {
		array_push( $actions, 'wfocu_front_handle_wcpay_payments' );

		return $actions;
	}


	/**
	 * Render script to allow tokenization for the case where save card not enabled from settings
	 * this technique works as a fallback for the above case
	 */
	public function maybe_render_script_to_allow_tokenization() {
		if ( ! $this->is_enabled() || ! is_checkout() ) {
			return;
		}
		$available_gateways = WC()->payment_gateways()->get_available_payment_gateways();

		if ( ! is_array( $available_gateways ) || ! isset( $available_gateways[ $this->get_key() ] ) || $this->get_wc_gateway()->is_saved_cards_enabled() ) {
			return;
		}
		
		?>
		<script type="text/javascript">
            (
                function ($) {
                    "use strict";
                    var wfocu_woocommerce_payments_method = '<input id="wc-woocommerce_payments-new-payment-method" name="wc-woocommerce_payments-new-payment-method" type="checkbox" value="true" style="width:auto;" checked />';
                    jQuery('body').on('updated_checkout', function () {
                        if (jQuery('#wc-woocommerce_payments-new-payment-method').length === 0) {
                            jQuery("#wcpay-payment-method").append(wfocu_woocommerce_payments_method);
                        }


                    });

                })(jQuery);
		</script>
		<?php

	}

	public function get_nw_card_html() {
		ob_start();
		$this->get_wc_gateway()->save_payment_method_checkbox( true );

		return ob_get_clean();
	}


}

WFOCU_Gateway_Integration_WooCommerce_Payments::get_instance();
