<?php

namespace App;

if(!defined('ABSPATH')){
	exit;
}

use App\ToType;
use JShrink\Minifier as JSMinifier;
use tubalmartin\CssMin\Minifier as CSSMinifier;

/**
 * AssetsWizard class, used to enqueue msb theme assets with wp_enqueue_scripts hook
 * 
 * @hook AssetsWizardStyles
 * 
 * @hook AssetsWizardScripts
 * 
 * @hook AssetsWizardStyle/{$asset_handle}
 * 
 * @hook AssetsWizardSscript/{$asset_handle}
 * 
 * @version 0.0.3
 * 
 * @author E-World
 */

final class AssetsWizard {

    /**
     * Method to sort assets by dependencies
     * 
     * @param array $assets
     * 
     * @return array $assets
     */
    private static function sort_assets($assets){

        if($assets && is_array($assets) && !empty($assets)){
            usort($assets, function($a, $b){
                return !in_array($a['handle'], $b['deps']);
            });
        }
        return $assets;
    }
    /**
     * Method to parse arguments passed to enqueue method 
     * and apply filters to them
     * 
     * @param array $args 
     * 
     * @param string 'style'|'script' $type
     * 
     * @return array $parsed_args
     */
    private static function parse_enqueue_args(array $args, string $type){
        switch ($type) {
            case 'style':
                $hookname = 'AssetsWizardStyle';
                break;
            case 'script':
                $hookname = 'AssetsWizardScript';
                break;
            default:
                return $args;
        }
        $parsed_args = [
            'minify' => (isset($args['minify']) && is_array($args['minify'])) ? [
                'handle' => $args['minify']['handle'] ?? NULL,
                'filepath' => (isset($args['path']) && isset($args['minify']['basename'])) ? $args['path'] . $args['minify']['basename'] : NULL,
                'src' => (isset($args['url']) && isset($args['minify']['basename'])) ?  $args['url'] . $args['minify']['basename'] : NULL,
                'deps' => $args['minify']['deps'] ?? [],
                'ver' => $args['ver'] ?? false
            ] : false,
            'assets' => apply_filters($hookname.'s', array_map(function($asset) use($args, $hookname){
                return apply_filters($hookname.'/'.$asset[0], [
                    'handle' => $asset[0],
                    'filepath' => isset($args['path']) ? $args['path'].$asset[1] : get_template_directory().'/'.$asset[1],
                    'src' => isset($args['url']) ? $args['url'].$asset[1] : get_template_directory_uri().'/'.$asset[1],
                    'deps' => $asset[2] ?? [],
                    'ver' => $args['ver'] ?? false
                ]);
            }, is_array($args['assets']) ? $args['assets'] : []))
        ];
        return $parsed_args;
    }

    /**
     * Method to get all assets content
     * 
     * @param array $assets
     * 
     * @return string $contents
     */
    private static function get_assets_content(array $assets){
        $contents = '';
        foreach($assets as $asset){
            if(file_exists($asset['filepath'])){
                $contents .= trim(file_get_contents($asset['filepath'])) . ' ';
            }
        }
        return $contents;
    }

    /**
     * Method to create mininfied file
     * 
     * @param string $assets_filepath  file to put contents
     * 
     * @param string $asset_content     content to put intro the file
     */
    private static function create_minified_asset(string $assets_filepath, string $asset_content){
        $asset = fopen($assets_filepath, "w") or die();
        fwrite($asset, $asset_content);
    }

    /**
     * Method to enqueue styles
     * 
     * @param array $args
     */
    public static function enqueue_styles(array $args){
        // parse arguments
        $args = self::parse_enqueue_args($args, 'style');
        if(!empty($args['assets'])){
            if($args['minify']['handle']){
                if(!file_exists($args['minify']['filepath'])){
                    // sort assets by deps
                    $args['assets'] = self::sort_assets($args['assets']);
                    // get all assets content
                    $assets_content = self::get_assets_content($args['assets']);
                    // minify assets content
                    $minified_assets_content = self::minify('css', $assets_content);
                    // create minified file
                    self::create_minified_asset($args['minify']['filepath'], $minified_assets_content);
                }
                // enqueue minified asset 
                wp_enqueue_style($args['minify']['handle'], $args['minify']['src'], $args['minify']['deps'], $args['minify']['ver']);
            } else {
                // enqueue all given assets 
                foreach($args['assets'] as $asset){
                    wp_enqueue_style($asset['handle'], $asset['src'], $asset['deps'], $asset['ver']);
                }
            }
        }
    }

    /**
     * Method to enqueue scripts
     * 
     * @param array $args
     */
    public static function enqueue_scripts(array $args){
        // parse arguments
        $args = self::parse_enqueue_args($args, 'script');
        if(!empty($args['assets'])){
            if($args['minify']['handle']){
                if(!file_exists($args['minify']['filepath'])){
                    // sort assets by deps
                    $args['assets'] = self::sort_assets($args['assets']);
                    // get all assets content
                    $assets_content = self::get_assets_content($args['assets']);
                    // minify assets content
                    $minified_assets_content = self::minify('js', $assets_content);
                    // create minified file
                    self::create_minified_asset($args['minify']['filepath'], $minified_assets_content);
                }
                // enqueue minified asset 
                wp_enqueue_script($args['minify']['handle'], $args['minify']['src'], $args['minify']['deps'], $args['minify']['ver'], true);
            } else {
                // enqueue all given assets 
                foreach($args['assets'] as $asset){
                    wp_enqueue_script($asset['handle'], $asset['src'], $asset['deps'], $asset['ver'], true);
                }
            }
        }
    }

    /**
     * Method to minify asset file
     * 
     * @param string $type              styles|scripts
     * @param string $content
     * @return string $minified         compressed asset content
     */
    private static function minify($type, $content){
        switch($type){
            case 'js':
                $minified = JSMinifier::minify($content, array('flaggedComments' => false));
                break;
            case 'css':
                $compressor = new CSSMinifier;
                $compressor->removeImportantComments();
                $minified = $compressor->run($content);
                break;
        }
        return $minified;
    }

    private function __construct(){}
    private function __clone(){}
    private function __wakeup(){}
}