<?php

namespace App;

if(!defined('ABSPATH')){
	exit;
}

use App\Acf;
use App\Storage;
use App\Rest;

class Bootstrap {

    /**
     * Method prevents post rewrite rules to apply
     * Used in admin area to prevent acf option pages
     * to be hidden due to rewrite rules bug
     */
    public static function flushRewriteRules(){
        if(Acf::get('post_category_slug_prefix', 'option', 'bool')){
			flush_rewrite_rules();
		}
    }

    /**
     * Method that adds post category prefixes in post permalinks
     * 
     * When admin adds post category in Theme Settings/Permalink Settings
     * for each post category theme adds category slug to post permalink
     * 
     * @source https://wordpress.stackexchange.com/a/228414
     */
    public static function addPostCategorySlugPrefix(){
		if(Acf::get('post_category_slug_prefix', 'option', 'bool')):
			foreach(Acf::get('post_category_slug_prefix', 'option', 'arr') as $category):
				if(is_object($category) && property_exists($category, 'slug')){
					$category_slug = $category->slug;
					// add rewrite rules
					add_rewrite_rule('^' . $category_slug . '/([^/]+)/?$', 'index.php?post_type=post&name=$matches[1]', 'top');
					// rewrite post links
					add_filter('post_link', function($post_link, $post, $leave_name = false, $sample = false) use($category_slug){
						if(!is_admin() && has_category($category_slug, $post)){
							$post_link = str_replace('/' . $post->post_name, '/' . $category_slug . '/' . $post->post_name, $post_link);
						}
						return $post_link;
					}, 10, 4);
				}
			endforeach;
		endif;
    }
    
    /**
     * Method to store current page object and id since WordPress
     * default post_id storage conflicts with pages like Woocommerce shop page 
     * since it is consider as product archive and page at the same time.
     * Also consider 404 and Search pages objects
     * 
     * @var WP_Post|WP_term     Storage queried_object key
     * 
     * @var int                 Storage queried_object_id key
     */
    public static function queriedObject(){
        // get object
        if(function_exists('is_shop') && is_shop()){
            $to_set = get_post(wc_get_page_id('shop'));
        } elseif(is_404() || is_category()){
            $to_set = msb_get_template_page('404');
        } elseif(is_search()){
            $to_set = msb_get_template_page('search');
        } else {
            $to_set = get_queried_object();
        }
        // set id
        if(is_a($to_set, 'WP_Post')){
            Storage::set('queried_object_id', $to_set->ID);
        } elseif(is_a($to_set, 'WP_Term')){
            Storage::set('queried_object_id', $to_set->term_id);
        }
        // set object
        Storage::set('queried_object', $to_set);
    }

    /**
     * Method adding theme support for
     * WooCommerce plugin, title tag, custom logo
     * and small thumbnail image size
     */
    public static function themeSupport(){
		add_theme_support('woocommerce');
		add_theme_support('title-tag');
		add_theme_support('custom-logo');
		add_image_size('small_thumbnail', 50, 9999, false);
    }

    /**
     * Register template navigation menus
     */
    public static function navigation(){
		register_nav_menus([
			'header_menu' => __('Header menu', 'msb'),
			'header_menu_mobile' => __('Header menu mobile', 'msb'),
			'footer_info' => __('Footer Info', 'msb'),
			'soical' => __('Social Links', 'msb'),
			'our_shops' => __('Our Shops', 'msb')
		]);
    }

    /**
     * Method to add theme custom REST endpoints and
     * add fields to existing WordPress endpoints
     * 
     * - shop page filter endpoint
     * - add to cart endpoint
     * - get cart endpoint
     * - comment post endpoint
     * - comment list endpoint
     * - search products endpoint
     * 
     * @see config.php
     * 
     * And add post image src field to post endpoint
     */
    public static function api(){
        Rest::massRegister(Storage::get('config', 'api'));
        register_rest_field(
            'post', 
            'featured_image', 
            array(
                'get_callback' => function($post){
                    return msb_get_image_url(get_post_thumbnail_id($post['id']), 'medium');
                }
            )
        );
    }

    /**
     * Report all errors and add debugging enabled notice
     */
    public static function debug(){
        error_reporting(E_ALL);
        ini_set('display_errors', 1);
        ini_set('display_startup_errors', 1);
        add_action('wp_head', function(){
            c_dump('MSB v.' . Storage::get('config', 'version') . ': debug enabled!', 'warn', 'font-weight: bold;');
        }, 1, 0);
    }
}