<?php

namespace App;

if(!defined('ABSPATH')){
	exit;
}

use App\Acf;
use WC_Order_Query;

class BootstrapDeliveredOrders {

    /**
     * Bind method to a specific hook then schedule it
     * And set order delifery confirmation checker
     */
    public static function actions(){
        add_action('msb_get_delivered_orders_and_send_emails', 'App\BootstrapDeliveredOrders::getDeliveredOrdersAndSendEmails');
        add_action('init', 'App\BootstrapDeliveredOrders::scheduleEvent');
        add_action('template_redirect', 'App\BootstrapDeliveredOrders::confirmDeliveredOrder');
    }

    /**
     * Schedule daily 'msb_get_delivered_orders_and_send_emails' binded action
     */
    public static function scheduleEvent(){
        if(Acf::get('delivered_order_enabled', 'option', 'bool')){
            if(!wp_next_scheduled('msb_get_delivered_orders_and_send_emails')){
                wp_schedule_event(time(), 'daily', 'msb_get_delivered_orders_and_send_emails');
            }
        } else {
            wp_clear_scheduled_hook('msb_get_delivered_orders_and_send_emails');
        }
    }

    /**
     * Method that gets possible delivered orders
     * and send emails with delivered confirmation link
     * 
     * This function is binded to 'msb_get_delivered_orders_and_send_emails' action
     * 
     * @since 0.1.1
     */
    public static function getDeliveredOrdersAndSendEmails(){
		// Adding custom meta value filter for completed orders 
		// pending for devlivered status
		add_filter('woocommerce_order_data_store_cpt_get_orders_query', function($query, $query_vars){
			if(isset($query_vars['pending_to_be_delivered'])){
				$query['meta_query'][] = array(
					'key' => 'pending_to_be_delivered',
					'compare' => boolval($query_vars['pending_to_be_delivered']) ? 'IN' : 'NOT EXISTS'
				);
			}
			return $query;
		}, 1, 2);
		// get days as unix to consider query parameters
		$days_as_unix = intval(get_field('order_consider_delivered_after', 'option')) * 86400;
		$up_to = time();
		$from = $up_to - $days_as_unix;
		// query orders
		$query = new WC_Order_Query(array(
			'limit' => 15,
			'date_paid' => $from . '...' . $up_to,
			'status' => 'completed',
			'pending_to_be_delivered' => 0
		));
		$orders = $query->get_orders();
		/**
		 * Loop orders sending emails and set meta value
		 * pending_to_be_delivered to exclude these orders
		 * from the next query
		 */
		for($i = 0; $i < count($orders); $i++){
			// message
			$message = Acf::get('order_delivery_conf_email_mess', 'option', 'str');
			// add confirmation link to email message
			$page_id = Acf::get('delivered_order_page', 'option', 'int');
			if($page_id){
				$page_url = get_permalink($page_id); 
				// add order key for confirmation
				$page_url_args = parse_url($page_url, PHP_URL_QUERY) ? '&' : '?';
				$page_url_args .= 'wc_order_key=' . $orders[$i]->get_order_key();
				$message .= '<p><a href="' . $page_url . $page_url_args .  '">' . get_the_title($page_id) . '</a></p>';
			}
			// add shortcode support
			$message = apply_filters('the_content', $message);
			// send the mail
			$mail = wp_mail(
				$order->get_billing_email(),
				Acf::get('order_delivery_conf_email_subj', 'option', 'str'), 
				$message, 
				array(
					'content-type: text/html',
					'From: ' . get_bloginfo('name') . ' <' . sanitize_email(get_field('company_email', 'option')) . '>',
					'Cc: ' . sanitize_email(get_field('company_email', 'option'))
				)
			);
			// set pending custom meta
			if(isset($mail) && $mail == true){
				update_post_meta($orders[$i]->get_id(), 'pending_to_be_delivered', 'yes');
			}
		}
    }

    /**
     * Confirm order is being delivered and maybe send rate us email if
     * if post id is same as delivered confimation page id and order key is correct
     * else the page will be cosnidered as 404
     */
    public static function confirmDeliveredOrder(){
        // check if is delivered order page
		if(Acf::get('delivered_order_enabled', 'option') && get_the_ID() == Acf::get('delivered_order_page', 'option', 'int')){
			// check wc_order_key
			$order_key = isset($_GET['wc_order_key']) ? esc_sql($_GET['wc_order_key']) : NULL;
			// get order by order key
			if(!is_null($order_key)){
				$order_id = wc_get_order_id_by_order_key($order_key);
				$order = wc_get_order($order_id);
			}
			// check if order exists and is completed or delivered
			if((isset($order) && is_a($order, 'WC_Order')) && ($order->get_status() == "completed" || $order->get_status() == "delivered")){
				if($order->get_status() == "completed"){
					// order is delivered
					$order->set_status('wc-delivered');
					$order->save();
					// order is no longer pending to be delivered
					delete_post_meta($order->get_id(), 'pending_to_be_delivered');
					// send email after confirmation
					if(Acf::get('delivered_order_after_enabled', 'option')){
						// mail message
						$message = Acf::get('order_delivery_conf_after_mess', 'option', 'str');
						// add page link to email message
						$page_id = Acf::get('delivered_after_confirmation_page', 'option', 'int');
						if($page_id){
							$message .= '<p><a href="' . get_permalink($page_id) .  '">' . get_the_title($page_id) . '</a></p>';
						}
						// add shortcode support
						$message = apply_filters('the_content', $message);
						// send after delivery confimation mail
						wp_mail(
							$order->get_billing_email(), 
							Acf::get('order_delivery_conf_after_subject', 'option', 'str'), 
							$message, 
							array(
								'content-type: text/html',
								'From: ' . get_bloginfo('name') . ' <' . sanitize_email(get_field('company_email', 'option')) . '>',
								'Cc: ' . sanitize_email(get_field('company_email', 'option'))
							)
						);
					}
				}
			} else {
				msb_set_404();
			}
		}
    }
}