<?php

namespace App;

if(!defined('ABSPATH')){
	exit;
}

use WP_Query;
use WC_Payment_Gateways;
use WC_Customer;
use WC_Cart;
use App\Acf;
use App\WooCommerceReadonlyRole;

class BootstrapWoocommerce {

	/**
	 * Filter product with chosen categories from related products
	 * 
	 * @param $related_posts
	 * @param $product_id
	 * 
	 * @return array $related_posts		array diff with excluded product ids
	 * 									where exluded are products with chose category
	 */
	public static function excludeRealatedProducts(array $related_posts, $product_id){
		$categories_to_exclude = Acf::get('exclude_related_categories', $product_id, 'arr');
		if($categories_to_exclude && class_exists('WooCommerce')){
			$exclude_ids = wc_get_products(array(
				'status'    => 'publish',
				'limit'     => -1,
				'category'  => array_map(function($category){
						return $category->slug;
					}, $categories_to_exclude),
				'return'    => 'ids',
			));
			return array_diff($related_posts, $exclude_ids);
		} else {
			return $related_posts;
		}
	}


    /**
     * Method to autocomplete all woocommerce orders but bank trasfer
     *
     * @param mixed $order_id
     *
     */
    public static function autocompleteOrders($order_id){
        if($order_id && class_exists('WooCommerce')){
         
        }
    }

	/**
	 * Method that adds icon to paymet method based on ACF Custom field image presence
	 * When admin adds image to Theme Settings/Woocommerce/Checkout/Payment Logos
	 * they will be added to payment method label html
	 * 
	 * @param string $icon_html
	 * @param string $gateway_id
	 * 
	 * @return string $icon_html		with added image html
	 */
	public static function addGatewayIcons($icon_html, $gateway_id){
		if(class_exists('WC_Payment_Gateways')):
			$image_src = NULL;
			switch ($gateway_id){
				case 'bacs': // Direct Bank Transfer
					$image_src = msb_get_image_url(Acf::get('checkout_direct_bank_transfer_logo', 'option', 'str'));
					break;
				case 'stripe': // Stripe
					$image_src = msb_get_image_url(Acf::get('checkout_stripe_logo', 'option', 'str'));
					break;
			}
			if($image_src){
				// get gateways instance
				$gateway_instance = WC_Payment_Gateways::instance();
				// get payment gateways
				$payment_gateways = $gateway_instance->payment_gateways();
				// check if gateway exists
				if(isset($payment_gateways[$gateway_id])){
					$payment_settings = $payment_gateways[$gateway_id];
					// set icon html
					$icon_html = '<img src="' . $image_src . '" alt="' . $payment_settings->title . '">';
				}
			}
		endif;
		return $icon_html;
	}

	/**
	 * Method that removes Billing ad Shipping prefixes 
	 * from required error fields
	 * 
	 * @param string $error_notice
	 * 
	 * @param string $field_label
	 * 
	 * @return string $error_notice 	notice without shipping and billing 
	 * 
	 */
	public static function removeCheckoutBillingAndShippinPrefixes(string $error_notice, string $field_label){
		$field_label = substr($field_label, strpos($field_label, " "));
		if(trim($field_label)){
			$error_notice = sprintf( __( '%s is a required field.', 'woocommerce' ), '<strong>' . esc_html( $field_label ) . '</strong>' );
			return $error_notice;
		} else {
			return '';
		}
	}

	/**
	 * Method that filters checkout fields:
	 * - replace field label
	 * - add required option
	 * - unset placeholders
	 * 
	 * uses wp_get_checkout_fields_formatted function based in includes/legacy.php
	 * 
	 * @param array $fields			woocommerce checkout fields
	 * 
	 * @return array $to_return		formatted fields
	 */
	public static function filterCheckoutFields(array $fields){
		$to_return = array();
		$format_options = array_filter(Acf::get('checkout_fields', 'option', 'arr'));
		foreach($fields as $key=>$value){
			$format_results = wp_get_checkout_fields_formatted($value, $format_options);
			$format_options = $format_results['options'];
			$to_return[$key] = $format_results['fields'];
		}
		return $to_return;
	}

	/**
	 * Method to unapprove all posted comments including 
	 * reviews, questions and post comments if posting user is not admin
	 * 
	 * @param array $commentdata
	 * 
	 * @return array $commentdata		data with unapproved status
	 */
	public static function unapprovePostedComments(array $commentdata){
		if((isset($_POST['type'])) && ($_POST['type'] != '')){
			$commentdata['comment_type'] = wp_filter_nohtml_kses($_POST['type']);
			if(!(current_user_can('editor') || current_user_can('administrator'))){
				$commentdata['comment_approved'] = 0;
			}
		}
		return $commentdata;
	}

	/**
	 * Method that registers order statuses:
	 * - replacement
	 * - replacement only device
	 * - delivered
	 */
	public static function addOrderStatuses(){
		if(function_exists('register_post_status') && function_exists('add_filter')):
			// replacement
			register_post_status('wc-replacement', array(
				'label'                     => _x( 'Replacement', 'Order status', 'woocommerce'),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				'label_count'               => _n_noop( 'Replaced <span class="count">(%s)</span>', 'Replaced <span class="count">(%s)</span>', 'woocommerce')
			));
			// device only
			register_post_status('wc-replace-device', array(
				'label'                     => _x( 'Replacement Device Only', 'Order status', 'woocommerce'),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				'label_count'               => _n_noop( 'Replaced Device Only <span class="count">(%s)</span>', 'Replaced Device Only <span class="count">(%s)</span>', 'woocommerce')
			));
			// delivered
			register_post_status('wc-delivered', array(
				'label'                     => _x( 'Delivered', 'Order status', 'woocommerce'),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				'label_count'               => _n_noop( 'Delivered <span class="count">(%s)</span>', 'Delivered <span class="count">(%s)</span>', 'woocommerce')
			));
			add_filter('wc_order_statuses', function($statuses){
				// replacement
				$statuses['wc-replacement'] = _x( 'Replacement', 'Order status', 'woocommerce' );
				// device only
				$statuses['wc-replace-device'] = _x( 'Replacement Device Only', 'Order status', 'woocommerce' );
				// delivered
				$statuses['wc-delivered'] = _x( 'Delivered', 'Order status', 'woocommerce' );

				return $statuses;
			});
		endif;
	}

	 
	/**
	 * Method to create customer support user role
	 * Uses WooCommerceReadonlyRole class
	 */
	public static function customerSupportUserRole(){
		if(function_exists('add_filter')):
			add_filter('wcror_role_details', function(){
				return [
					'slug' => 'msb_customer_support',
					'name' => 'Customer Support'
				];
			});
			$woocommerce_readonly_role = new WooCommerceReadonlyRole();
		endif;
	}

	/**
	 * Method that adds woocommerce states by adding them to custom fields
	 * For example when admin adds state Theme Settings/Woocommerce/Checkout States
	 * function replaces standart woocommerce region states with the ones admin added
	 * 
	 * @param array $states			woocommerce sandart states
	 * 
	 * @return array $states 		states with added items
	 * 
	 * @since 0.1.1
	 */
	public static function addWoocommerceStates(array $states){
		if(class_exists('WooCommerce')):
			$locale_keys = array_keys(WC()->countries->get_country_locale());
			$states_to_push = Acf::get('checkout_states', 'option', 'arr');
			if($states_to_push && $locale_keys){
				foreach($locale_keys as $key){
					// remove standart states
					unset($states[$key]);
					// add custom ones
					foreach($states_to_push as $state){
						$states[$key][$state['key']] = $state['label'];
					}
				}
			}
		endif;
	  return $states;
	}
	
    /**
     * Method to activate cart enviorment, mainly used
     * in theme custom rest endpoints
     */
    public static function cart(){
		if(class_exists('WooCommerce')){
			// include and activate cart enviorment
			include_once WC_ABSPATH . 'includes/wc-cart-functions.php';
			include_once WC_ABSPATH . 'includes/wc-notice-functions.php';
			if(null === WC()->session){
				$session_class = apply_filters('woocommerce_session_handler', 'WC_Session_Handler');
				WC()->session = new $session_class();
				WC()->session->init();
			}
			if (null === WC()->customer){
				WC()->customer = new WC_Customer(get_current_user_id(), true);
			}
			if(null === WC()->cart){
				WC()->cart = new WC_Cart();
				WC()->cart->get_cart();
			}
		}
    }
}