<?php

namespace App;

if(!defined('ABSPATH')){
	exit;
}

use WP_Query;
use App\Storage;
use App\Acf;
use App\ToType;
use App\WCBooster;

class Product {

    /**
     * Props data mainly used as caching tool
     * 
     * @var array
     */
    private $props = array();

    /**
     * Method to set prop
     * 
     * @param string $key
     * 
     * @param mixed $value
     */
    protected function set_prop(string $key, $value){
        $this->props[$key] = $value;
    }

    /**
     * Method to set prop
     * 
     * @param string $key
     * 
     * @return mixed|NULL       NULL if key not found
     */
    protected function get_prop(string $key){
        return $this->props[$key] ?? NULL;
    }

    /**
     * Method to get bestseller option
     * 
     * @return boolean
     */
    public function is_bestseller(){
        if(is_null($this->get_prop('is_bestseller'))){
            $product = $this->get_product();
            $to_set = Acf::get('bestseller_option', $product->get_id(), 'bool');
            $this->set_prop('is_bestseller', $to_set);
        }
        return $this->get_prop('is_bestseller');
    }

    /**
     * Method to get regular price
     * 
     * @param boolean $formatted
     * 
     * @return string
     */
    public function get_regular_price($formatted = false){
        $product = $this->get_product();
        if($product->is_type('variable')){
            $price = $product->get_variation_regular_price('min');
        } elseif($product->is_type('simple')) {

            $price = $product->get_regular_price();
        }
        return $formatted ? wc_price($price) : $price;
    }

    /**
     * Method to get sale price
     * 
     * @param boolean $formatted
     * 
     * @return string
     */
    public function get_sale_price($formatted = false){
        $product = $this->get_product();
        if($product->is_type('variable')){
            $price = $product->get_variation_sale_price('min');
        } else {
            $price = $product->get_price();
        }
        return $formatted ? wc_price($price) : $price;
    }

    /**
     * Method to get save price (Regular Price - Sale Price)
     * 
     * @param boolean $formatted
     * 
     * @return string
     */
    public function get_save_price($formatted = false){
        $price = intval($this->get_regular_price()) - intval($this->get_sale_price());
        return $formatted ? wc_price($price) : $price;
    }

    /**
     * Method to get average rating
     * If fake is chosen method will return the value
     * of the custom field if exists else will return
     * real value
     * 
     * @param boolean $fake
     * 
     * @return float
     */
    public function get_average_rating($fake = true){
        $fake_postfix = $fake ? '_fake' : '';
        if(!$this->get_prop('product_average_rating'.$fake_postfix)){
            $product = $this->get_product();
            if($fake){
                $fake_rating = Acf::get('average_rating', $product->get_id(), 'float');
                if($fake_rating){
                    $rating = $fake_rating;
                } else {
                    $rating = $product->get_average_rating();
                }
            } else {
                $rating = $product->get_average_rating();
            }
            $this->set_prop('product_average_rating'.$fake_postfix, number_format($rating, 2));
        }
        return $this->get_prop('product_average_rating'.$fake_postfix);
    }
     
    /**
     * Reviews Count method will return fake reviews count
     * if fake reviews are more than real ones else will return
     * real product reviews count
     * 
     * @return int
     */
    public function get_reviews_count(){
        if(!$this->get_prop('product_reviews_count')){
            $product_id = $this->get_product()->get_id();
            $reviews_count = $this->get_product()->get_review_count();
            $fake_reviews_count = Acf::get('reviews_count', $product_id, 'int');
            if($fake_reviews_count > $reviews_count){
                $reviews_count = $fake_reviews_count;
            }
            $this->set_prop('product_reviews_count', $reviews_count);
        }
        return $this->get_prop('product_reviews_count');
    }

    /**
     * Method to get product questions count
     * 
     * @return int
     */
    public function get_questins_count(){
        if(!$this->get_prop('product_questins_count')){
            $this->set_prop(
                'product_questins_count', 
                get_comments(array(
                    'post_id' => $this->get_product()->get_id(),
                    'type__in' => 'question',
                    'hierarchical' => false,
                    'status' => 'approve',
                    'count' => true
                ))
            );
        }
        return $this->get_prop('product_questins_count');
    }

    /**
     * Get Featured Image Url
     * 
     * @return string
     */
    public function get_featured_image_url($size = 'full'){
        if(!$this->get_prop('product_featured_image_url')){
            $post_id = $this->get_prop('product')->get_id();
            $attachemnt_id = get_post_thumbnail_id($post_id);
            $image_url = ToType::evaluate(wp_get_attachment_image_src($attachemnt_id, $size)[0], 'string');
            $this->set_prop('product_featured_image_url', $image_url);
        }
        return $this->get_prop('product_featured_image_url');
    }
    /**
     * Product Pretty Permalink
     * 
     * @return string
     */
    public function get_permalink(){
        $permalink = '';
        if(function_exists('get_permalink')){
            $permalink = get_permalink($this->get_product()->get_id());
        }
        return $permalink;
    }

    /**
     * Ideal For Categories
     * 
     * @return array            array of product_cat terms
     */
    public function get_ideal_for(){
        return $this->get_product_categories('attr');
    }

    /**
     * Frequency Categories
     * 
     * @return array            array of product_cat terms
     */
    public function get_frequency(){
        if(!$this->get_prop('product_cat_frequency_result')){
            $categories = $this->get_product_categories('frequency');
            if($categories && isset($categories[0])){
                $this->set_prop('product_cat_frequency_result', $categories[0]->name);
            } else {
                $this->set_prop('product_cat_frequency_result', '');
            }
        }
        return $this->get_prop('product_cat_frequency_result');
    }

    /**
     * Provider Categories
     * 
     * @return array            array of product_cat terms
     */
    public function get_providers(){
        return $this->get_product_categories('providers');
    }

    /**
     * Coverage Categories
     * 
     * @return array            array of product_cat terms
     */
    public function get_coverage(){
        if(!$this->get_prop('product_cat_coverage_result')){
            $coverages = $this->get_product_categories('coverage');
            $coverage = !empty($coverages) ? end($coverages)->name : '';
            $this->set_prop('product_cat_coverage_result', $coverage);
        }
        return $this->get_prop('product_cat_coverage_result');
    }

    /**
     * Product GSM
     * Get all GSM terms of a product, explode their names into
     * words array, filters unique and return imploded string
     * 
     * @return string
     */
    public function get_gsm(){
        if(!$this->get_prop('product_cat_gsm_result')){
            $gsm_categories = $this->get_product_categories('gsm');
            if(!empty($gsm_categories)){
                $unique_words = array();
                $seen_the_word = array();
                $delimeter = " & ";
                $delimeter_set = false;
                // loop categories
                for($i = 0; $i < count($gsm_categories); $i++){
                    // get words
                    $words = explode(" ", $gsm_categories[$i]->name);
                    // loop words
                    foreach($words as $word){
                        // identify delimeter
                        if(strlen($word) == 1 || strpos($word, ";")){
                            if($delimeter_set == false){
                                $delimeter = " " . $word . " ";
                            }
                        // in case of not delimeter
                        } elseif(!isset($seen_the_word[$word])){
                            $seen_the_word[$word] = true;
                            $unique_words[] = preg_replace("/[^a-zA-Z0-9]+/", "", $word);
                        }
                    }
                }
                $this->set_prop('product_cat_gsm_result', implode($delimeter, array_unique($unique_words)));
            }
        }
        return $this->get_prop('product_cat_gsm_result');
    }

    /**
     * Band Property exists
     * 
     * @return boolean
     */
    public function has_band(){
        return !empty($this->get_product_categories('band'));
    }

    /**
     * Band Property name
     * 
     * @return string
     */
    public function get_band_name(){
        if($this->has_band()){
            $band = $this->get_product_categories('band')[0] ?? NULL;
            if(is_object($band)){
                return $this->get_product_categories('band')[0]->name;
            }
        }
        return '';
    }

    /**
     * Band Property color
     * 
     * @return string
     */
    public function get_band_color(){
        if($this->has_band()){
            $band = $this->get_product_categories('band')[0] ?? NULL;
            if(is_object($band)){
                $band_id = $this->get_product_categories('band')[0]->term_id;
                return get_option('cat_background_color'.$band_id);
            }
        }
        return '';
    }

    /**
     * Units sold count
     * Add fake sold count by adding reviews count value multiplied by 3
     * 
     * @return int
     */
    public function get_units_sold(){
        if(!$this->get_prop('units_sold')){
            $to_set = $this->get_product()->get_total_sales();
            if($this->get_reviews_count()){
                $to_set = $to_set + ($this->get_reviews_count() * 3);
            }
            $this->set_prop('units_sold', $to_set);
        }
        return $this->get_prop('units_sold');
    }

    /**
     * Json variations, used to pass variation info to
     * products page javascript in order to impliment
     * custom logic as attribute image|desciption and 
     * impossible variations handle
     * 
     * @return string
     */
    public function get_json_variations(){
        if($this->get_prop('product')->is_type('variable')){
            if(!$this->get_prop('json_variations')){
                $product = $this->get_prop('product');
                $to_set = array();
                foreach($product->get_available_variations() as $variation){
                    if($variation['variation_is_active']){
                        $to_set[$variation['variation_id']] = array(
                            "query" => http_build_query($variation['attributes']),
                            "regular_price" => $variation['display_regular_price'],
                            "sale_price" => $variation['display_price'],
                            "is_on_sale" => ($variation['display_regular_price'] != $variation['display_price'])
                        );
                    }
                }
                $this->set_prop('json_variations', json_encode($to_set));
            }
            return $this->get_prop('json_variations');
        } else {
            return NULL;
        }
    }

    /**
     * Product Carriers
     * 
     * @return array            array of product_cat terms
     */
    public function get_default_carriers(){
        if(is_null($this->get_prop('carriers'))){
            $product_id = $this->wc()->get_id();
            if(Acf::get('carriers_option', $product_id, 'bool')){
                $to_set = Acf::get('msb_product_carriers', 'option', 'arr');
            } else {
                $to_set = array();
            }
            $this->set_prop('carriers', $to_set);
        }
        return $this->get_prop('carriers');
    }

    /**
     * Get product attributes with custom fields
     * Will return empty array if product is not variable
     * 
     * Example output:
     * [
     *  'key' => 'pa_warrany',
     *  'label' => 'Warranty'
     *  'terms' => [
     *      [
     *          'name' => '2 Years'
     *          'slug' => '2-years',
     *          'image' => 'http://example.com/image.png',
     *          'description' => 'We provide 2 Years warranty on all our products'
     *      ]
     *  ]
     * ]
     * 
     * @return Generator|array
     */
    public function get_variation_attributes(){
        if($this->wc()->is_type('variable') && $this->wc()->get_attributes()){
            $default_attributes = $this->wc()->get_default_attributes();
            foreach($this->wc()->get_attributes() as $key=>$term){
                if($term['variation'] == true && $term['options']){
                    $to_yield = [
                        'key' => $key,
                        'label' => esc_attr(wc_attribute_label($key)),
                        'terms' => []
                    ];
                    foreach(wc_get_product_terms($this->wc()->get_id(), $key) as $index => $attribute_term){
                        $to_push = [
                            "name" => esc_attr($attribute_term->name),
                            "slug" => $attribute_term->slug,
                            "description" => esc_attr($attribute_term->description),
                            "image" => msb_get_image_url(Acf::get('thumb', $key . '_' . $attribute_term->term_id), 'medium')
                        ];
                        if(isset($_GET['attribute_' . $key])){
                            $to_push["chosen"] = sanitize_text_field($_GET['attribute_' . $key]) == $attribute_term->slug;
                        } else {
                            $to_push["chosen"] = isset($default_attributes[$key]) ? $default_attributes[$key] == $attribute_term->slug : $index == 0;
                        }
                        $to_yield['terms'][] = $to_push;
                    }
                    yield $to_yield;
                }
            }
        } else {
            return [];
        }
    }

    /**
     * Query products, 
     * 
     * @param array $args           arguments passed to WP_Query
     *                              object constructor
     * 
     * @return Generator|array      generator of App/Product instances
     */
    public static function query(array $args = []){
        $query = new WP_Query(array_merge([
            'orderby' => 'menu_order',
            'order' => 'ASC',
            'tax_query' => [],
            'posts_per_page' => -1,
            'post_status' => 'publish'
        ], $args, [
            'fields' => 'ids',
            'cache_results' => true,
            'update_post_meta_cache' => true,
            'post_type' => 'product',
        ]));
        if($query->posts){
            foreach($query->posts as $post_id){
                yield new Product($post_id);
            }
        } else {
            return [];
        }
    }

    /**
     * Get product categories using WCBooster class and
     * cache them
     * 
     * @param string $key
     * 
     * @return array                array of product_cat terms
     */
    private function get_product_categories($key){
        if(!$this->get_prop('product_cat_' . $key)){
            $WC_booster_instance = $this->get_prop('wc_booster');
            $categories = $WC_booster_instance->wc_get_product_categories($this->get_product(), $key);
            $this->set_prop('product_cat_'. $key, $categories);
        }
        return $this->get_prop('product_cat_' . $key);
    }

    /**
     * Set product as WC_Product instance
     * 
     * @param int|WC_Product $product
     */
    private function set_product($product){
        if(is_a($product, 'WC_Product')){
            $this->set_prop('product', $product);
        } elseif(intval($product)){
            $this->set_prop('product', wc_get_product(intval($product)));
        }
    }

    /**
     * Get product as WC_Product instance
     * 
     * @return WC_Product
     */
    private function get_product(){
        return $this->get_prop('product');
    }

    /**
     * Get product as WC_Product instance,
     * used in public
     * 
     * @return WC_Product
     */
    public function wc(){
        return $this->get_product();
    }

    /**
     * Set WCBooster instance with all booster parameters
     * 
     * @param WCBooster
     */
    private function set_wc_booster_instance(WCBooster $wc_booster){
        $this->set_prop('wc_booster', $wc_booster);
    }

    function __construct($product){
        $this->set_product($product);
        $this->set_wc_booster_instance(Storage::get('booster_props'));
    }
}