<?php

namespace App\Rest;

if(!defined('ABSPATH')){
	exit;
}

use WP_Query;
use WP_REST_Request;
use App\AbstractRest;
use App\ToType;
use App\Validate;
use App\Sanitize;
use App\Acf;
use App\Product;

class GetProducts extends AbstractRest {

    public static function getProduct($msb_product){
        return array(
            "id" => $msb_product->wc()->get_id(),
            "name" => $msb_product->wc()->get_name(),
            "permalink" => $msb_product->wc()->get_permalink(),
            "featured_image" => $msb_product->get_featured_image_url('medium'),
            "reviews_allowed" => $msb_product->wc()->get_reviews_allowed(),
            "rating_count" => $msb_product->get_reviews_count(),
            "average_rating" => $msb_product->get_average_rating(),
            "coverage" => $msb_product->get_coverage(),
            "band" => $msb_product->has_band() ? array(
                'color' => $msb_product->get_band_color(),
                'name' => $msb_product->get_band_name()
            ) : array(),
            "gsm" => $msb_product->get_gsm(),
            "attributes" => self::getProductCategories($msb_product->get_ideal_for()),
            "providers" => self::getProductCategories($msb_product->get_providers()),
            "on_sale" => $msb_product->wc()->is_on_sale(),
            "regular_price" => $msb_product->get_regular_price(true),
            "sale_price" => $msb_product->get_sale_price(true),
            "save_price" => $msb_product->get_save_price(true),
            "is_bestseller" => $msb_product->is_bestseller()
        );
    }

    public static function getProductCategories($categories_arr){
        $to_return = array();
        foreach($categories_arr as $category){
            $image_id =  get_term_meta($category->term_id, 'thumbnail_id', true);
            $to_return[] = array(
                "image" => msb_get_image_url($image_id, 'small_thumbnail'),
                "name" => $category->name
            );
        }
        return $to_return;
    }


    public static function init(){
        $instance = self::getInstance();
        $instance->method = 'GET';
        $instance->namespace = 'msb/v2/product';
        $instance->route = '/list';

        /**
         * Query filter arguments
         */
        // categories in
        $instance->setArg('categories', array(
            'required' => null,
            'default' => array(),
            'validate_callback' => function($categories){
                return is_array($categories) && empty($categories) || Validate::check($categories, 'arr/int');
            },
            'sanitize_callback' => function($categories){
                return Sanitize::get($categories, 'int');
            }
        ));
        // categories exclude
        $instance->setArg('categories_not_in', array(
            'required' => null,
            'default' => array(),
            'validate_callback' => function($categories_not_in){
                return is_array($categories_not_in) && empty($categories_not_in) || Validate::check($categories_not_in, 'arr/int');
            },
            'sanitize_callback' => function($categories_not_in){
                return Sanitize::get($categories_not_in, 'int');
            }
        ));
        
        /**
         * Pagincation arguments
         */
        // products per page
        $instance->setArg('per_page', array(
            'required' => null,
            'default' => 10,
            'validate_callback' => function($per_page){
                return Validate::check($per_page, 'int');
            },
            'sanitize_callback' => function($per_page){
                return ToType::evaluate($per_page, 'int');
            }
        ));
        // page number
        $instance->setArg('page', array(
            'required' => null,
            'default' => 1,
            'validate_callback' => function($page){
                return Validate::check($page, 'int');
            },
            'sanitize_callback' => function($page){
                return ToType::evaluate($page, 'int');
            }
        ));


        // set callback function
        $instance->setCallback(function(WP_REST_Request $request){
           // query products arguments
           $query_args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => $request->get_param('per_page'),
            'paged' => $request->get_param('page'),
            'tax_query' => [
                array(
                    'taxonomy'      => 'product_cat',
                    'terms'         => $request->get_param('categories'),
                    'field'         => 'term_id',
                    'operator'      => 'AND',
                    'include_children' => false
                ),
                array(
                    'taxonomy'      => 'product_cat',
                    'terms'         => $request->get_param('categories_not_in'),
                    'field'         => 'term_id',
                    'operator'      => 'NOT IN',
                    'include_children' => false
                ),
                array(
                    'taxonomy' => 'product_visibility',
                    'field'    => 'name',
                    'terms'    => array('exclude-from-catalog'),
                    'operator' => 'NOT IN',
                    'include_children' => false,
                )
            ]
        );
        // loop query
        $products = [];
        foreach(Product::query($query_args) as $product){
            $products[] = self::getProduct($product);
        }
        return $products;
    });
    return $instance;
    }
}