<?php

namespace App;

if(!defined('ABSPATH')){
	exit;
}

use Closure;
use App\ToType;

class Section {

    /**
     * Path to get templates
     * 
     * @var string
     */
    private $template_path = 'template-parts/sections/';

    /**
     * Template name, must be same as php template filename
     * 
     * 
     * @var string
     */
    private $template_name = '';

    /**
     * Template data
     * 
     * @var array
     */
    private $data = [];

    /**
     * Template functions
     * 
     * @var array
     */
    private $functions = [];

    /**
     * Get template filepath and add filter name equal to it
     * 
     * @return string $to_return
     */
    public function getTemplateFilepath(){
        $to_return = $this->template_path.$this->template_name.'.php';
        return apply_filters($to_return, get_template_directory().'/'.$to_return);
    }

    /**
     * Get template functions filepath and add filter name equal to it
     * 
     * @return string $to_return
     */
    public function getTemplateFunctionsFilepath(){
        $to_return = $this->template_path.$this->template_name.'.functions.php';
        return apply_filters($to_return, get_template_directory().'/'.$to_return);
    }

    /**
     * Set template data, if key is not set
     * affect whole section data
     * 
     * @param mixed $data
     * 
     * @param string $key
     */
    public function setData($data, string $key = NULL){
        if(is_null($key)){
            $this->data = $data;
        } elseif(is_string($key)){
            $this->data[$key] = $this->data;
        }
    }

    /**
     * Method to access section data
     * 
     * @param string $type          used to evaluate variable
     * 
     * @param string $wp_filter     used to apply WordPress filters
     * 
     * @return string|mixed $to_return
     */
    public function getData(string $key = NULL, string $type = NULL, string $wp_filter = NULL){
        $to_return = '';
        if(!is_null($key)){
            if(isset($this->data[$key])){
                $to_return = $this->data[$key];
                if(!is_null($type)){
                    $to_return = ToType::evaluate($to_return, $type);
                }
                if(!is_null($wp_filter)){
                    $to_return = apply_filters($wp_filter, $to_return);
                }
            }
        } else {
            $to_return = $this->data;
        }
        return $to_return;
    }

    /**
     * Method to set template functions
     * 
     * @param string $key
     * 
     * @param Closure $func
     */
    public function setFunc(string $key, Closure $func){
        $this->functions[$key] = $func;
    }

    /**
     * Method to invoke template function
     * 
     * @param string $key
     * 
     * @param array $args
     */
    public function callFunc(string $key, array $args = []){
        if(isset($this->functions[$key])){
            return call_user_func($this->functions[$key], ...$args);
        }
    }

    /**
     * Method to get and require section template parts
     * including template and its helper functions
     */
    private function getTemplate(){
        if(file_exists($this->getTemplateFunctionsFilepath())){
            require $this->getTemplateFunctionsFilepath();
        }
        if(file_exists($this->getTemplateFilepath())){
            require $this->getTemplateFilepath();
        }
    }

    /**
     * Method rendering section
     */
    public function render(){
        $this->getTemplate();
    }

    /**
     * Method that gets all class properties
     * 
     * @return array 
     */
    public function get(){
        return [];
    }

    function __construct(string $name, array $data = [], string $path = 'template-parts/sections/'){
        $this->template_name = $name;
        $this->setData($data);
        $this->template_path = $path;
    }
}