<?php

namespace App;

if(!defined('ABSPATH')){
	exit;
}

/**
 * Class to handle Woocommerce product special properties assigned
 * to product by product_cat taxonomy. For example by_use, providers and others.
 */
class WCBooster {

	/**
	 * Parsed product_cat properties
	 * 
	 * @var array
	 */
	private $categories = [];

	/**
	 * Setup product categories
	 * 
	 * @param array $props
	 */
	function __construct(array $props){
		$this->setCategoriesAttribute($props['categories'] ?? []);
		add_filter('terms_clauses', [$this, 'addMultipleParentsQuery'], 10, 3);
	}

	/**
	 * Adding term_parents argument to that adds query by multiple
	 * term parents
	 * 
	 * @param array $pieces
	 * 
	 * @param array $taxonomies
	 * 
	 * @param array $args
	 * 
	 * 
	 * @return array $pieces
	 */
	public function addMultipleParentsQuery($pieces, $taxonomies, $args){
		if(in_array('product_cat', $taxonomies) && isset($args['term_parents'])){
			if((!isset($args['child_of']) || !$args['child_of']) && (!isset($args['child_of']) || !$args['parent'])){
				$where = [];
				foreach($args['term_parents'] as $parent){
					$parent = intval($parent);
					if($parent){
						$where[] = " tt.parent = '$parent'";
					}
				}
				if($where){
					$pieces['where'] .= " AND ( ".implode(' OR ', $where)." )";
				}
			}
		}
		return $pieces;
	}

	/**
	 * Get product category terms by parent slug
	 * previously set in class constructor.
	 * 
	 * If key is null return categories class property
	 * 
	 * @param string|NULL $key
	 * 
	 * @return array $terms|$this->categories
	 */
	private function getProductCategories(string $key = NULL){
		if(is_null($key)){
			return $this->categories;
		} elseif(isset($this->categories[$key]) && $this->categories[$key]['parents']){
			if(is_null($this->categories[$key]['categories'])){
				$this->categories[$key]['categories'] = [];
				$terms = get_terms([
					'taxonomy' => 'product_cat',
					'term_parents' => $this->categories[$key]['parents']
				]);
				if($terms){
					foreach($terms as $term){
						$this->categories[$key]['categories'][$term->term_id] = $term;
					}
				}
			}
			return $this->categories[$key]['categories'];
		} else {
			return [];
		}
	}
	

	/**
	 * Set product categories data
	 * 
	 * @param array $categories
	 */
	private function setCategoriesAttribute(array $categories){
		foreach($categories as $key=>$data){
			if(isset($data['label']) && isset($data['parents'])){
				$this->categories[$key] = [
					'label' => $data['label'],
					'parents' => $data['parents'],
					'categories' => NULL
				];
			}
		}
	}

	/**
	 * Get categories stack label
	 * 
	 * @param string $key
	 * 
	 * @return string
	 */
	public function wc_get_categories_label($key){
		return $this->categories[$key]['label'] ?? '';
	}

	/**
	 * Get categories of a parent or all or specific category stack
	 * 
	 * @param string $parent_category
	 * 
	 * @return array $terms
	 */
	public function wc_get_categories(string $parent_category = NULL){
		return did_action('woocommerce_init') ? $this->getProductCategories($parent_category) : [];
	}
	
	/**
	 * Method to get product categories of a parent
	 * 
	 * @param WC_Product|mixed $product
	 * 
	 * @param string $parent_category
	 * 
	 * @return array $categories_to_return
	 */
	public function wc_get_product_categories($product, string $parent_category){
		$categories_to_return = [];
		if(did_action('woocommerce_init')){
			$child_categories = $this->getProductCategories($parent_category);
			if($child_categories):
				$wc_product = is_a($product, 'WC_product') ? $product : wc_get_product($product);
				if($wc_product && $wc_product->get_category_ids()){
					foreach($wc_product->get_category_ids() as $category_id):
						if(isset($child_categories[$category_id])){
							$categories_to_return[] = $child_categories[$category_id];
						}
					endforeach;
				}
			endif;
		}
		return $categories_to_return;
	}
}