<?php

if(!defined('ABSPATH')){
	exit;
}

use App\Storage;
use App\SectionFactory;
use App\Product;

/**
 * Function to get template post object
 * 
 * @param string $template_name
 * 
 * @return WP_Post|NULL
 */
if(!function_exists('msb_get_template_page')):
	function msb_get_template_page(string $template_name){
		$template_pages = get_pages([
			'meta_key' => '_wp_page_template',
			'meta_value' => $template_name . '.php'
		]);
		if(is_array($template_pages) && !empty($template_pages)){
			return $template_pages[0];
		} else {
			return NULL;
		}
	}
endif;

/**
 * Function to get nav menu term by location name
 * 
 * @param string $location
 * 
 * @return WP_Term|false
 */
if(!function_exists('msb_get_menu_by_location')):
	function msb_get_menu_by_location(string $location){
		$locations = get_nav_menu_locations();
		if($location && isset($locations[$location])){
			return get_term($locations[$location], 'nav_menu');
		}
		return false;
	}
  endif;

/**
 * Section function to get or render page sections
 * 
 * @param int $post_id
 * @param bool $render
 * 
 * @return array $sections
 */
if(!function_exists('msb_get_sections')):
	function msb_get_sections($post_id = NULL, bool $render = true){
		$sections = new SectionFactory;
		$sections->post_id = is_null($post_id) ? Storage::get('queried_object') : $post_id;
		if($render){
			$sections->render();
		} else {
			return $sections->get();
		}
	}
endif;

/**
 * Function retrieving theme product object
 * You also can access wc product class using wc() method
 * For example $msb_product->wc()->get_id()
 * 
 * @param int|object $product       can be product id order WC_Product instance
 * 
 * @return object $msb_product
 * 
 * @since 0.1.1
 * 
 */

if(!function_exists('msb_get_product')):
    function msb_get_product($product){
        return new Product($product);
    }
endif;

/**
 * Function retrieving a list of App\Product objects
 * 
 * @param array $query_args			arguments you normally pass to WP_Query object
 * 
 * @return array $msb_product      	array of msb_products
 * 
 * @since 0.1.1
 * 
 */

if(!function_exists('msb_get_products')):
    function msb_get_products($query_args){
		return Product::query($query_args);
    }
endif;

/**
 * Function to get image src by attachemnt id
 * 
 * @param string|int $attachment_id
 * @param string $size
 * 
 * @return string $image_src		image src or empty string
 */
if(!function_exists('msb_get_image_url')):
	function msb_get_image_url($attachment_id, string $size = 'full'){
		if($attachment_id){
			return wp_get_attachment_image_src($attachment_id, $size)[0] ?? '';
		}
		return '';
	}
endif;

/**
 * Function to set 404 headers and force
 * 404 template part to render
 */
if(!function_exists('msb_set_404')):
	function msb_set_404(){
		global $wp_query;
		if($wp_query){
			$wp_query->set_404();
			status_header(404);
			get_template_part(404); 
			exit();
		}
	}
endif;

/**
 * Function to get product variation attributes in array format
 * used in cart and checkout templates to render cart item attributes
 * 
 * If taxonomy not exists also include customly added 
 * taxonomy created by theme such as Product Carriers
 * 
 * @param array $product_attributes		product attibutes extracted with cart item
 * 										variations key or with order get_meta_data method
 * 
 * @return array $to_return				array of associative arrays
 * 										with label and value keys
 * 
 * @since 0.1.2
 */
if(!function_exists('msb_get_cart_item_attributes')):
	function msb_get_cart_item_attributes(array $product_attributes){
		$to_return = [];
		if(class_exists('WooCommerce')):
			foreach($product_attributes as $key=>$value){
				// extract needles from WC_Meta_Data object
				if(is_a($value, 'WC_Meta_Data')){
					$item_data = $value->get_data();
					$attribute_key = $item_data['key'] ?? '';
					$attribute_value = $item_data['value'] ?? '';
					$taxonomy_name = $attribute_key;
				// extract needles from cart item variations key value
				} else {
					$attribute_key = $key;
					$attribute_value = $value;
					$taxonomy_name = wc_attribute_taxonomy_name(str_replace('attribute_pa_', '', urldecode($attribute_key)));
				}
				// get taxonomy term
				$term = get_term_by('slug', $attribute_value, $taxonomy_name);
				if(!is_wp_error($term) && $term){
					$hidden_option = get_field('is_hidden', $term);
					if(is_bool($hidden_option) && $hidden_option == false){
						$to_return[] = array(
							'label' => wc_attribute_label($taxonomy_name),
							'value' => $term->name
						);
					}
				} else {
					$to_return[] = array(
						'label' => str_replace('-', ' ', $attribute_key),
						'value' => $attribute_value
					);
				}
			}
		endif;
		return $to_return;
	}
endif;

/**
 * Fucntion to dump the variable with <pre> tags for pretty view.
 * If 'die' string is passed die function is called
 * 
 * @param spread $var
 */
if(!function_exists('v_dump')):
	function v_dump(...$var){
		if(func_num_args()){
			foreach(func_get_args() as $var){
				echo '<pre>';
				if(is_callable($var)){
					var_dump(call_user_func($var));
				} elseif($var == 'die'){
					die();
				} else {
					var_dump($var);
				}
				echo '</pre>';
			}
		} else {
			echo '<pre>';
			var_dump(123);
			echo '</pre>';
			die();
		}
	}
endif;

/**
 * Function to console log the variable
 * 
 * @param mixed $var
 * 
 * @param string $type
 * 
 * @param string $css
 */
if(!function_exists('c_dump')):
	function c_dump($var, string $type = 'log', string $css = ''){
		echo "<script>console.$type(";
		if($css){
			$var = '%c' . $var;
		}
		if(is_callable($var)){
			echo json_encode(call_user_func($var));
		} else {
			echo json_encode($var);
		}
		if($css){
			echo ", " . json_encode($css);
		}
		echo ");</script>";
	}
endif;